//*****************************************************************************
//
//! @file am_vos_audio_buffer.c
//!
//! @brief Header file of am_vos_audio_buffer.c
//
//*****************************************************************************

//*****************************************************************************
//
// Copyright (c) 2024, Ambiq Micro, Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the copyright holder nor the names of its
// contributors may be used to endorse or promote products derived from this
// software without specific prior written permission.
//
// Third party software included in this distribution is subject to the
// additional license terms as defined in the /docs/licenses directory.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// This is part of revision ambiqvos_r4.5-fdfa8cf6a4 of the AmbiqSuite Development Package.
//
//*****************************************************************************

#ifndef AM_VOS_AUDIO_BUFFER_H
#define AM_VOS_AUDIO_BUFFER_H

#include <stdint.h>
#include <stdbool.h>
#include <string.h>

#include "am_vos_sys_config.h"

//*****************************************************************************
//
// Ring buffer setup structure definitions
//
//*****************************************************************************
typedef struct
{
    int32_t indx;
    volatile uint8_t* pData;
    volatile uint32_t ui32ByteSize;
}am_vos_ringbuff_setup_t;

//*****************************************************************************
//
// Ring buffer structure definitions
//
//*****************************************************************************
typedef struct
{
    volatile uint8_t *pui8Data;
    volatile uint32_t ui32BufferTail_write;
    volatile uint32_t ui32BufferHead_read;
    volatile uint32_t ui32OverWriting;
    volatile uint32_t ui32Capacity;
}am_vos_ring_buffer_t;

//
// Read/Head combo enum name
//
typedef enum{
    AM_AUDIO_BUFFER_STEREO,             // raw audio data from 2 channels of mics
    AM_AUDIO_BUFFER_MONO,               // audio data filtered by dspc signal chain
    AM_AUDIO_BUFFER_ENCODED,            // compressed audio data
//    AM_AUDIO_BUFFER_COMM,
    AM_AUDIO_BUFFER_MAX
}enum_audio_buffer_t;

typedef struct{
    uint8_t * head;
    uint8_t * tail;
}audio_buffer_ptr_t;

#if (configUSE_Sensory_THF && configUSE_PREROLL)
#define BYTES_UNIVERSAL_BUFFER_SIZE (96 * 1024)
#elif configUSE_AMVOS_ATVV
#define BYTES_UNIVERSAL_BUFFER_SIZE (48 * 1024)    // 4KB for mSBC, 6KB for OPUS as a minimal buffer size.
#else
#define BYTES_UNIVERSAL_BUFFER_SIZE (32 * 1024)    // 4KB for mSBC, 6KB for OPUS as a minimal buffer size.
#endif // configUSE_Sensory_THF && configUSE_PREROLL


//*****************************************************************************
//
// Function-like macros.
//
//*****************************************************************************
//#define am_vos_ring_buffer_empty(psBuffer)                                   \
//    (((psBuffer)->ui32BufferTail_write == (psBuffer)->ui32BufferHead_read) && ((psBuffer)->ui32OverWriting == 0))
//
//#define am_vos_ring_buffer_full(psBuffer)                                    \
//    (((psBuffer)->ui32BufferTail_write == (psBuffer)->ui32BufferHead_read) && ((psBuffer)->ui32OverWriting == 1))
//
//#define am_vos_ring_buffer_overwrite(psBuffer)                                    \
//    (((psBuffer)->ui32BufferTail_write != (psBuffer)->ui32BufferHead_read) && ((psBuffer)->ui32OverWriting == 1))

//*****************************************************************************
//
// External function definitions
//
//*****************************************************************************

void am_vos_ring_buffer_init(am_vos_ring_buffer_t* ring_buffs, am_vos_ringbuff_setup_t setup);
void am_vos_ring_buffer_init_all(am_vos_ring_buffer_t* ring_buffs, const am_vos_ringbuff_setup_t* setup_array, uint32_t ui32BufferCount);
uint32_t am_vos_ring_buffer_push(am_vos_ring_buffer_t *psBuffer, void *pvSource, uint32_t ui32Bytes, bool bFullCheck);
uint32_t am_vos_ring_buffer_pop(am_vos_ring_buffer_t *psBuffer, void *pvDest,
                         uint32_t ui32Bytes);
uint32_t am_vos_get_ring_buffer_status(am_vos_ring_buffer_t *psBuffer);
uint8_t am_vos_ring_buffer_empty(am_vos_ring_buffer_t *psBuffer);
void am_vos_flush_ring_buffer(am_vos_ring_buffer_t *psBuffer);
uint32_t am_vos_ring_process(am_vos_ring_buffer_t *psSource, void *pvDest, uint32_t process_frame_bytes);

extern void am_audio_buffer_init(void);
extern bool am_audio_buffer_push(enum_audio_buffer_t buffer_type, void *pvSource, uint32_t ui32Bytes);
extern uint32_t am_audio_buffer_pop(enum_audio_buffer_t buffer_type, void *pvDest, uint32_t ui32Bytes);
extern uint32_t am_audio_buffer_rewind (enum_audio_buffer_t buffer_type, uint32_t offset_back);
void am_audio_buffer_nested_push(enum_audio_buffer_t buffer_push_to, void *pvSource, uint32_t ui32Bytes);
void am_audio_buffer_nested_pop(enum_audio_buffer_t buffer_pop_from, void *pvDest, uint32_t ui32Bytes);
bool am_audio_universal_buffer_status_check(am_vos_ring_buffer_t* nested_buff_list);

#endif // AM_VOS_AUDIO_BUFFER_H
