//*****************************************************************************
//
//! @file am_vos_logic.c
//!
//! @brief LED indication for system status, wakeword detection
//
//*****************************************************************************

//*****************************************************************************
//
// Copyright (c) 2024, Ambiq Micro, Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the copyright holder nor the names of its
// contributors may be used to endorse or promote products derived from this
// software without specific prior written permission.
//
// Third party software included in this distribution is subject to the
// additional license terms as defined in the /docs/licenses directory.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// This is part of revision ambiqvos_r4.5-fdfa8cf6a4 of the AmbiqSuite Development Package.
//
//*****************************************************************************

#include "am_vos_sys_config.h"
#include "am_vos_board_setup.h"

#include "am_util.h"
#include "am_devices_led.h"
#include "am_vos_utils.h"

#include "am_vos_task.h"
#include "am_vos_init.h"
#include "am_vos_logic.h"

#if configUSE_LEDs
void am_vos_logic_led_all_on(void)
{
    //LEDs on
    am_devices_led_array_out(am_bsp_psLEDs, AM_BSP_NUM_LEDS, 0xFF);
}

void am_vos_logic_led_all_off(void)
{
    //LEDs off
    am_devices_led_array_out(am_bsp_psLEDs, AM_BSP_NUM_LEDS, 0x0);
}

void am_vos_logic_cmd_led_on(int8_t i8CmdIndex)
{
    am_vos_logic_led_all_off();

    if(i8CmdIndex == -1)
      return;

    if(i8CmdIndex < AM_BSP_NUM_LEDS)
        am_devices_led_on(am_bsp_psLEDs, i8CmdIndex);
    else
        am_devices_led_on(am_bsp_psLEDs, i8CmdIndex % AM_BSP_NUM_LEDS);
}

void am_vos_logic_mcmd_led_on(int8_t i8On)
{
    am_vos_logic_led_all_off();

    if(i8On == 0)
      return;

    for (int i = 0; i < AM_BSP_NUM_LEDS; i++) {
        if (i8On & (0x1 << i))
            am_devices_led_on(am_bsp_psLEDs, i);
    }
}

void am_vos_logic_led_swirl(uint8_t type)
{
    TickType_t xDelaySwirl = pdMS_TO_TICKS(50);

    am_vos_logic_led_all_off();
    switch(type)
    {
        case 0: // keyword detection (Swirling)
            for(int i = 0; i < (AM_BSP_NUM_LEDS * 2); i++)
            {
                am_devices_led_toggle(am_bsp_psLEDs, i % AM_BSP_NUM_LEDS);
                vTaskDelay(xDelaySwirl);
            }
            break;
        case 1: // power up (Swirling twice)
            for(int i = 0; i < (AM_BSP_NUM_LEDS * 4); i++)
            {
                am_devices_led_toggle(am_bsp_psLEDs, i % AM_BSP_NUM_LEDS);
                am_util_delay_ms(50);
            }
            break;
        case 2: // blinking all LEDs twice (for Maya)
            for(int i = 0; i < (AM_BSP_NUM_LEDS * 2); i++)
            {
                if(i == AM_BSP_NUM_LEDS)
                {
                    vTaskDelay(xDelaySwirl * 2);
                    am_vos_logic_led_all_off();
                    vTaskDelay(xDelaySwirl * 2);
                }
                am_devices_led_toggle(am_bsp_psLEDs, i % AM_BSP_NUM_LEDS);
            }
            vTaskDelay(xDelaySwirl);

            break;
        case 3:  // Alternate Keyword detect (Reverse Swirling)
            for(int i = (AM_BSP_NUM_LEDS * 2)-1; i >= 0; i--)
            {
                am_devices_led_toggle(am_bsp_psLEDs, i % AM_BSP_NUM_LEDS);
                vTaskDelay(xDelaySwirl);
            }
            break;

        default:
            // do nothing
            break;
    }
    am_vos_logic_led_all_off();
}
#endif

