//*****************************************************************************
//
//! @file am_vos_utils.h
//!
//! @brief Task, IPC defines
//
//*****************************************************************************

//*****************************************************************************
//
// Copyright (c) 2024, Ambiq Micro, Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the copyright holder nor the names of its
// contributors may be used to endorse or promote products derived from this
// software without specific prior written permission.
//
// Third party software included in this distribution is subject to the
// additional license terms as defined in the /docs/licenses directory.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// This is part of revision ambiqvos_r4.5-fdfa8cf6a4 of the AmbiqSuite Development Package.
//
//*****************************************************************************

#ifndef AM_VOS_UTILS_H
#define AM_VOS_UTILS_H

#include <stdint.h>
#include <stdbool.h>
//*****************************************************************************
//
// FreeRTOS include files.
//
//*****************************************************************************
#include "FreeRTOS.h"
#include "task.h"
#include "queue.h"
#include "timers.h"

//*****************************************************************************
//
// Application utils include files.
//
//*****************************************************************************
#include "am_vos_audio_buffer.h"

/**************************************************************************************************
 Type Macros
**************************************************************************************************/

// System message macro definition 
#define EMPTY_MESSAGE                   1
#define HEARTBEAT_TIMER_MESSAGE         2
#define DOUBLE_TAP_MESSAGE              3
#define MUTE_MIC_MESSAGE                4
#define PUSH_TO_TALK_MESSAGE            5
#define PAIRING_MODE_MESSAGE            6
#define RTT_AMU2S_MESSAGE               7
#define KEY_WORD_GOT_MESSAGE            0xaa111155

#define AM_VOS_STDIO_BUFFNUM                        8
#define AM_VOS_PRINTF_BUFFSIZE                      256     // has to be 256, there are long prints... 
#define AM_VOS_PRINTF_TOTAL_BUFFSIZE                (AM_VOS_STDIO_BUFFNUM*AM_VOS_PRINTF_BUFFSIZE) 

typedef struct
{
    TaskHandle_t task;
    QueueHandle_t queue;
//  am_vos_task_enum_t parent;
} am_vos_task_t;

//*****************************************************************************
//
// App task list structure typedefs
//
//*****************************************************************************
// this enum list contains all app related application tasks between AM_TASK_NONE and AM_MAX_TASK
// where each index corresponds to a TASK or ISR.
// Naming convention for a task: AM_TASK_xxx
// Naming convention for an ISR: AM_ISR_xxx
typedef enum
{
    AM_VOS_TASK_NONE = 0, // The enum must begin with this value as named.
    AM_VOS_TASK_LED,

#if configUSE_LOG_UART0 || configUSE_PRINTF_UART0
    AM_VOS_TASK_UART0,
#endif // configUSE_LOG_UART0 || configUSE_PRINTF_UART0

#if configUSE_STDIO_PRINTF
    AM_VOS_TASK_STDIO,
#endif // configUSE_STDIO_PRINTF

    AM_VOS_TASK_VAD,

    AM_VOS_TASK_AUD_PROCESSING,

#if configUSE_MODEL_INFERENCE
    AM_VOS_TASK_MODEL_INFERENCE,
#endif // configUSE_MODEL_INFERENCE

    AM_VOS_TASK_AWE_TICK,

#if configUSE_BLE
    AM_VOS_BLE,
#endif // configUSE_BLE

#if configUSE_AUDIO_CODEC
    AM_VOS_TASK_CODEC,
#endif // configUSE_AUDIO_CODEC

    AM_VOS_ISR_GPIO,
    AM_VOS_ISR_UART0,
    
#if USE_DMIC_PDM
    AM_VOS_ISR_PDM,
#endif // USE_DMIC_PDM
    
#if USE_AMIC_AUDADC
    AM_VOS_ISR_ADC,
#endif // USE_AMIC_AUDADC
    
    AM_SLEEP,
    AM_VOS_MAX_TASK // The enum must end with this value as named.
} am_vos_task_enum_t;

//*****************************************************************************
//
// App timer list structure typedefs
//
//*****************************************************************************
typedef enum
{
    AM_VOS_TIMER_NONE = 0,      // The enum must begin with this value as named.
    AM_VOS_TIMER_HEART_BEAT,
    AM_VOS_TIMER_GSENSOR_PERIOD,
    AM_VOS_TIMER_LONG_PRESS,    // Timer for long press
    AM_VOS_MAX_TIMER            // The enum must end with this value as named.
} am_vos_timer_enum_t;

typedef struct
{
    //Task Setup
    am_vos_task_enum_t indx; //Specify this task's index.
    
    TaskFunction_t pxTaskCode; //FreeRTOS function pointer to task
    const char* const pcName; // FreeRTOS name
    const uint16_t usStackDepth; // Stack Size
    void * const pvParameters; // FreeRTOS task parameter mechanism
    UBaseType_t uxPriority; // FreeRTOS Task Priority
    
    //Queue Setup
    const UBaseType_t uxQueueLength;
    //const UBaseType_t uxItemSize; //sizeof (queue_element)
    
    //Parent Task
//    am_vos_task_enum_t parent; //Specify parent task.
}am_vos_task_setup_t;

//******************************************************************
// System timer setup structure
//
//******************************************************************

typedef struct
{
    //Task Setup
    am_vos_timer_enum_t   indx; //Specify this task's index.
    const char* const           pcTimerName; // FreeRTOS name
    TickType_t                  xTimerPeriodInTicks;
    UBaseType_t                 uxAutoReload; 
    TimerCallbackFunction_t     pxTaskCode; //FreeRTOS function pointer to task
}am_vos_timer_setup_t;

//
// QUEUE entry for all corresponding queues
// Structure definition of the message element in queue
//
typedef struct
{
    am_vos_task_enum_t    Source; // The sender lets the receiver know the source.
    uint32_t                    ui32MessageType; // May be redefined per task to index different uses of *pData.
    union{
        uint32_t ui32Note;      // short message for simple communication between tasks
        uint32_t ui32Length;    // data length for long message
        uint32_t ui32Indx;      // index for printf string buffer
    }info;
    am_vos_ring_buffer_t*       pDataBuffer; 

}am_vos_task_queue_element_t;

typedef void (*am_vos_stdio_printf_t)(uint32_t indx, uint8_t print_type);

typedef struct _AmVosUtil {
    am_vos_stdio_printf_t         pfnPrintf;                                      // function pointer for printf
    uint32_t                            ui32BuffIndx;
    char                                pcStdioBuff[AM_VOS_PRINTF_TOTAL_BUFFSIZE];      // buffer for printf
    
    uint8_t                             pui8RingBuffer[BYTES_UNIVERSAL_BUFFER_SIZE];
    am_vos_ring_buffer_t                sRingBuf[AM_AUDIO_BUFFER_MAX];
} AmVosUtil;

extern AmVosUtil g_sAmUtil; 

void am_vos_stdio_printf_init(am_vos_stdio_printf_t pfnPrintFunc);
uint32_t am_vos_stdio_printf(uint8_t print_type, const char *pcFmt, ...);

extern am_vos_task_t am_KWD_tasks[AM_VOS_MAX_TASK];
extern TimerHandle_t am_KWD_timers[AM_VOS_MAX_TIMER];
extern void am_vos_task_init(void);
extern void am_vos_task_create(am_vos_task_setup_t setup);
extern bool am_vos_task_send(am_vos_task_enum_t Source, am_vos_task_enum_t Dest, uint32_t ui32MessageType, uint32_t ui32Notice, am_vos_ring_buffer_t *pData);
extern bool am_vos_task_send_fromISR(am_vos_task_enum_t Source, am_vos_task_enum_t Dest, uint32_t ui32MessageType, uint32_t ui32Notice, am_vos_ring_buffer_t *pData);
//extern bool am_vos_task_send_up(am_vos_task_enum_t Source, uint32_t MessageType, void *pData);
//extern bool am_vos_task_send_up_fromISR(am_vos_task_enum_t Source, uint32_t MessageType, void *pData);
extern bool am_vos_task_read(am_vos_task_enum_t indx, am_vos_task_queue_element_t *Element);
extern TaskHandle_t am_vos_task_get_task_handle(am_vos_task_enum_t indx);
extern void am_vos_task_create_all_tasks(const am_vos_task_setup_t *task_array, uint8_t task_count);
extern void am_vos_timer_create_all_timers(const am_vos_timer_setup_t *setup_array, uint8_t timer_count);
extern void am_vos_task_suspend(am_vos_task_enum_t indx);
extern void am_vos_task_resume(am_vos_task_enum_t indx);

#endif // AM_VOS_UTILS_H
