// ****************************************************************************
//
//  central_past_main.c
//! @file
//!
//! @brief Central PAST sample application.
//!
//! @{
//
// ****************************************************************************

//*****************************************************************************
//
// Copyright (c) 2026, Ambiq Micro, Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the copyright holder nor the names of its
// contributors may be used to endorse or promote products derived from this
// software without specific prior written permission.
//
// Third party software included in this distribution is subject to the
// additional license terms as defined in the /docs/licenses directory.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// This is part of revision stable-e24d618f43 of the AmbiqSuite Development Package.
//
//*****************************************************************************

#include <string.h>
#include "wsf_types.h"
#include "util/bstream.h"
#include "wsf_msg.h"
#include "wsf_trace.h"
#include "wsf_assert.h"
#include "wsf_buf.h"
#include "hci_api.h"
#include "dm_api.h"
#include "dm_priv.h"
#include "gap/gap_api.h"
#include "att_api.h"
#include "smp_api.h"
#include "app_cfg.h"
#include "app_api.h"
#include "app_db.h"
#include "app_ui.h"
#include "svc_core.h"
#include "svc_ch.h"
#include "svc_cte.h"
#include "gatt/gatt_api.h"
#include "atpc/atpc_api.h"
#include "central_past_api.h"
#include "util/calc128.h"
#include "app_main.h"
#include "am_util.h"

/**************************************************************************************************
  Macros
**************************************************************************************************/

/*! CentralPast app minimum length. */
#define CENTRAL_PAST_CTE_MIN_LEN               10

/*! CTE buttion actions. */
#define CENTRAL_PAST_CTE_ACTION_ENABLE         0
#define CENTRAL_PAST_CTE_ACTION_DISABLE        1

/*! CTE Interval. */
#define CENTRAL_PAST_CTE_INTERVAL              25

/*! CTE type. */
#define CENTRAL_PAST_CTE_TYPE                  HCI_CTE_TYPE_REQ_AOA

#define EXT_SCAN_NAME_MAX_LEN                  253

/**************************************************************************************************
  Local Variables
**************************************************************************************************/

/*! application control block */
struct
{
  uint16_t          hdlList[DM_CONN_MAX][APP_DB_HDL_LIST_LEN];   /*! Cached handle list */
  wsfHandlerId_t    handlerId;                      /*! WSF hander ID */
  bool_t            scanning;                       /*! TRUE if scanning */
  bool_t            autoConnect;                    /*! TRUE if auto-connecting */
  uint8_t           discState[DM_CONN_MAX];         /*! Service discovery state */
  uint8_t           hdlListLen;                     /*! Cached handle list length */
  uint8_t           btnConnId;                      /*! The index of the connection ID for button presses */
  uint8_t           cteAction[DM_CONN_MAX];         /*! CTE button action */
} centralPastCb;

/*! connection control block */
typedef struct
{
  appDbHdl_t          dbHdl;                        /*! Device database record handle type */
  uint8_t             addrType;                     /*! Type of address of device to connect to */
  bdAddr_t            addr;                         /*! Address of device to connect to */
  bool_t              doConnect;                    /*! TRUE to issue connect on scan complete */
  uint8_t             cteState;                     /*! State of CTE config process */
  uint8_t             secPhy;                       /*! Secondary Advertising channel from extended advertising rerport event*/
  uint16_t            eventType;                    /*! event type from extended advertising report event*/
#if (BT_53)
#if defined(BLE_PAST)
  bool_t              synced;                       /*! TRUE to indidate already synced the periodic advertising device */
  uint8_t             conn_id;                      /*! Connection identifier */
  dmSyncId_t          syncId;                       /*! Sync identifier */
  uint8_t             advSid;                       /*! Advertising SID of synced advertising device */
  uint8_t             advAddrType;                  /*! Address type of synced advertising device */
  bdAddr_t            advAddr;                      /*! Address of synced advertising device */
#endif // defined(BLE_PAST)
#endif // BT_53
} centralPastConnInfo_t;

centralPastConnInfo_t centralPastConnInfo;

/*! Identifiers for antenna */
static uint8_t centralPastAntennaIds[] = {0, 1};

/**************************************************************************************************
  Configurable Parameters
**************************************************************************************************/

/*! configurable parameters for master extendec scan */
static const appExtMasterCfg_t centralPastMasterExtCfg =
{
  {96},                                      /*! The scan interval, in 0.625 ms units */
  {48},                                      /*! The scan window, in 0.625 ms units  */
  0,                                         /*! The scan duration in ms */
  0,                                         /*! Scan period*/
  DM_DISC_MODE_NONE,                       /*! The GAP discovery mode */
  {DM_SCAN_TYPE_ACTIVE}                      /*! The scan type (active or passive) */
};

/*! configurable parameters for security */
static const appSecCfg_t centralPastSecCfg =
{
  DM_AUTH_BOND_FLAG | DM_AUTH_SC_FLAG,    /*! Authentication and bonding flags */
  DM_KEY_DIST_IRK,                        /*! Initiator key distribution flags */
  DM_KEY_DIST_LTK | DM_KEY_DIST_IRK,      /*! Responder key distribution flags */
  FALSE,                                  /*! TRUE if Out-of-band pairing data is present */
  FALSE                                    /*! TRUE to initiate security upon connection */
};

/*! TRUE if Out-of-band pairing data is to be sent */
static const bool_t centralPastSendOobData = FALSE;

/*! SMP security parameter configuration */
static const smpCfg_t centralPastSmpCfg =
{
  500,                                    /*! 'Repeated attempts' timeout in msec */
  SMP_IO_NO_IN_NO_OUT,                    /*! I/O Capability */
  7,                                      /*! Minimum encryption key length */
  16,                                     /*! Maximum encryption key length */
  1,                                      /*! Attempts to trigger 'repeated attempts' timeout */
  0,                                      /*! Device authentication requirements */
  64000,                                  /*! Maximum repeated attempts timeout in msec */
  64000,                                  /*! Time msec before attemptExp decreases */
  2                                       /*! Repeated attempts multiplier exponent */
};

/*! Connection parameters */
static const hciConnSpec_t centralPastConnCfg =
{
  40,                                     /*! Minimum connection interval in 1.25ms units */
  40,                                     /*! Maximum connection interval in 1.25ms units */
  0,                                      /*! Connection latency */
  600,                                    /*! Supervision timeout in 10ms units */
  0,                                      /*! Unused */
  0                                       /*! Unused */
};

/*! Configurable parameters for service and characteristic discovery */
static const appDiscCfg_t centralPastDiscCfg =
{
  FALSE,                                  /*! TRUE to wait for a secure connection before initiating discovery */
  TRUE                                    /*! TRUE to fall back on database hash to verify handles when no bond exists. */
};

static const appCfg_t centralPastAppCfg =
{
  FALSE,                                  /*! TRUE to abort service discovery if service not found */
  TRUE                                    /*! TRUE to disconnect if ATT transaction times out */
};

/*! ATT configurable parameters (increase MTU) */
static const attCfg_t centralPastAttCfg =
{
  15,                               /* ATT server service discovery connection idle timeout in seconds */
  23,                               /* desired ATT MTU */
  ATT_MAX_TRANS_TIMEOUT,            /* transcation timeout in seconds */
  4                                 /* number of queued prepare writes supported by server */
};

/*! local IRK */
static uint8_t localIrk[] =
{
  0xA6, 0xD9, 0xFF, 0x70, 0xD6, 0x1E, 0xF0, 0xA4, 0x46, 0x5F, 0x8D, 0x68, 0x19, 0xF3, 0xB4, 0x96
};

/**************************************************************************************************
  ATT Client Discovery Data
**************************************************************************************************/

/*! Discovery states:  enumeration of services to be discovered */
enum
{
  CENTRAL_PAST_DISC_GATT_SVC,      /*! GATT service */
  CENTRAL_PAST_DISC_GAP_SVC,       /*! GAP service */
  CENTRAL_PAST_DISC_CTE_SVC,       /*! Constant Tone Extension */
  CENTRAL_PAST_DISC_SVC_MAX        /*! Discovery complete */
};

/*! the Client handle list, centralPastCb.hdlList[], is set as follows:
 *
 *  ------------------------------- <- CENTRAL_PAST_DISC_GATT_START
 *  | GATT svc changed handle     |
 *  -------------------------------
 *  | GATT svc changed ccc handle |
 *  ------------------------------- <- CENTRAL_PAST_DISC_GAP_START
 *  | GAP central addr res handle |
 *  -------------------------------
 *  | GAP RPA Only handle         |
 *  ------------------------------- <- CENTRAL_PAST_DISC_CTE_START
 *  | CTE handles                 |
 *  | ...                         |
 *  -------------------------------
 */

/*! Start of each service's handles in the the handle list */
#define CENTRAL_PAST_DISC_GATT_START       0
#define CENTRAL_PAST_DISC_GAP_START        (CENTRAL_PAST_DISC_GATT_START + GATT_HDL_LIST_LEN)
#define CENTRAL_PAST_DISC_CTE_START        (CENTRAL_PAST_DISC_GAP_START + GAP_HDL_LIST_LEN)
#define CENTRAL_PAST_DISC_HDL_LIST_LEN     (CENTRAL_PAST_DISC_CTE_START + ATPC_CTE_HDL_LIST_LEN)

/*! Pointers into handle list for each service's handles */
static uint16_t *pCentralPastGattHdlList[DM_CONN_MAX];
static uint16_t *pCentralPastGapHdlList[DM_CONN_MAX];
static uint16_t *pCentralPastCteHdlList[DM_CONN_MAX];

/*! LESC OOB configuration */
static dmSecLescOobCfg_t *centralPastOobCfg;

/**************************************************************************************************
  ATT Client Configuration Data
**************************************************************************************************/

/*! Default value for CCC indications */
const uint8_t centralPastCccIndVal[2] = {UINT16_TO_BYTES(ATT_CLIENT_CFG_INDICATE)};

/*! Default value for CCC notifications */
const uint8_t centralPastCccNtfVal[2] = {UINT16_TO_BYTES(ATT_CLIENT_CFG_NOTIFY)};

/*! Default value for Client Supported Features (enable Robust Caching) */
const uint8_t centralPastCsfVal[1] = {ATTS_CSF_ROBUST_CACHING};

/*! List of characteristics to configure after service discovery */
static const attcDiscCfg_t centralPastDiscCfgList[] =
{
  /* Write:  GATT service changed ccc descriptor */
  {centralPastCccIndVal, sizeof(centralPastCccIndVal), (GATT_SC_CCC_HDL_IDX + CENTRAL_PAST_DISC_GATT_START)},

  /* Write: GATT clinet supported features */
  {centralPastCsfVal, sizeof(centralPastCsfVal), (GATT_CSF_HDL_IDX + CENTRAL_PAST_DISC_GATT_START)}
};

/*! Characteristic configuration list length */
#define CENTRAL_PAST_DISC_CFG_LIST_LEN   (sizeof(centralPastDiscCfgList) / sizeof(attcDiscCfg_t))

/*! sanity check:  make sure configuration list length is <= handle list length */
WSF_CT_ASSERT(CENTRAL_PAST_DISC_CFG_LIST_LEN <= CENTRAL_PAST_DISC_HDL_LIST_LEN);

/*************************************************************************************************/
/*!
 *  \brief  Application DM callback.
 *
 *  \param  pDmEvt  DM callback event
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralPastDmCback(dmEvt_t *pDmEvt)
{
  dmEvt_t   *pMsg;
  uint16_t  len;
  uint16_t  reportLen = 0;

  if (pDmEvt->hdr.event == DM_SEC_ECC_KEY_IND)
  {
    DmSecSetEccKey(&pDmEvt->eccMsg.data.key);

    /* If the local device sends OOB data. */
    if (centralPastSendOobData)
    {
      uint8_t oobLocalRandom[SMP_RAND_LEN];
      SecRand(oobLocalRandom, SMP_RAND_LEN);
      DmSecCalcOobReq(oobLocalRandom, pDmEvt->eccMsg.data.key.pubKey_x);
    }
  }
  else if (pDmEvt->hdr.event == DM_SEC_CALC_OOB_IND)
  {
    if (centralPastOobCfg == NULL)
    {
      centralPastOobCfg = WsfBufAlloc(sizeof(dmSecLescOobCfg_t));
    }

    if (centralPastOobCfg)
    {
      Calc128Cpy(centralPastOobCfg->localConfirm, pDmEvt->oobCalcInd.confirm);
      Calc128Cpy(centralPastOobCfg->localRandom, pDmEvt->oobCalcInd.random);
    }
  }
  else
  {
    len = DmSizeOfEvt(pDmEvt);

    if (pDmEvt->hdr.event == DM_EXT_SCAN_REPORT_IND)
    {
      reportLen = pDmEvt->extScanReport.len;
    }
    else if (pDmEvt->hdr.event == DM_PER_ADV_REPORT_IND)
    {
      reportLen = pDmEvt->perAdvReport.len;
      if (*(pDmEvt->perAdvReport.pData + 1) == DM_ADV_TYPE_LOCAL_NAME)
      {
        char rx_adv_data[128] = {0};
        memcpy(rx_adv_data, (char *)(pDmEvt->perAdvReport.pData + 2), pDmEvt->perAdvReport.len - 2);
        APP_TRACE_INFO2("[PER_ADV_REPORT] RSSI: %d, data: %s", pDmEvt->perAdvReport.rssi, rx_adv_data);
      }
      else
      {
        APP_TRACE_INFO2("[PER_ADV_REPORT] RSSI: %d, data length: %d", pDmEvt->perAdvReport.rssi, pDmEvt->perAdvReport.len);
      }
    }
    else if (pDmEvt->hdr.event == DM_CONN_IQ_REPORT_IND)
    {
      reportLen = pDmEvt->connIQReport.sampleCnt * 2;
    }
    else if (pDmEvt->hdr.event == DM_CONNLESS_IQ_REPORT_IND)
    {
      reportLen = pDmEvt->connlessIQReport.sampleCnt * 2;
    }
    #if (BT_53)
    #if defined(BLE_PAST)
    else if (pDmEvt->hdr.event == DM_PER_ADV_SYNC_LOST_IND)
    {
      centralPastConnInfo.synced = FALSE;
      AppExtScanStart(HCI_SCAN_PHY_LE_1M_BIT, centralPastMasterExtCfg.discMode,
                centralPastMasterExtCfg.scanType, 0, centralPastMasterExtCfg.scanPeriod);
    }
    else if ( pDmEvt->hdr.event == DM_PER_ADV_SYNC_EST_IND )
    {
      APP_TRACE_INFO0("Periodic advertising sync established");
      centralPastConnInfo.syncId = pDmEvt->hdr.param;
      if ( centralPastConnInfo.conn_id != 0 )
      {
        uint16_t service_data = 0x0000;
        DmPastSyncTrsf(centralPastConnInfo.conn_id, service_data, centralPastConnInfo.syncId);
      }

      AppExtScanStop();
    }
    #endif // defined(BLE_PAST)
    #endif // BT_53
    else
    {
      reportLen = 0;
    }

    if ((pMsg = WsfMsgAlloc(len + reportLen)) != NULL)
    {
      memcpy((uint8_t *)pMsg, (uint8_t *)pDmEvt, len);

      if (pDmEvt->hdr.event == DM_EXT_SCAN_REPORT_IND)
      {
        pMsg->extScanReport.pData = (uint8_t *)((uint8_t *)pMsg + len);
        memcpy(pMsg->extScanReport.pData, pDmEvt->extScanReport.pData, reportLen);
      }
      else if (pDmEvt->hdr.event == DM_PER_ADV_REPORT_IND)
      {
        pMsg->perAdvReport.pData = (uint8_t *)((uint8_t *)pMsg + len);
        memcpy(pMsg->perAdvReport.pData, pDmEvt->perAdvReport.pData, reportLen);
      }
      else if (pDmEvt->hdr.event == DM_CONN_IQ_REPORT_IND)
      {
        /* Copy I samples to space after end of report struct */
        pMsg->connIQReport.pISample = (int8_t *)((uint8_t *) pMsg + len);
        memcpy(pMsg->connIQReport.pISample, pDmEvt->connIQReport.pISample, pDmEvt->connIQReport.sampleCnt);

        /* Copy Q samples to space after I samples space */
        pMsg->connIQReport.pQSample = (int8_t *)((uint8_t *) pMsg->connIQReport.pISample + pDmEvt->connIQReport.sampleCnt);
        memcpy(pMsg->connIQReport.pQSample, pDmEvt->connIQReport.pQSample, pDmEvt->connIQReport.sampleCnt);
      }
      else if (pDmEvt->hdr.event == DM_CONNLESS_IQ_REPORT_IND)
      {
        /* Copy I samples to space after end of report struct */
        pMsg->connlessIQReport.pISample = (int8_t *)((uint8_t *) pMsg + len);
        memcpy(pMsg->connlessIQReport.pISample, pDmEvt->connlessIQReport.pISample, pDmEvt->connlessIQReport.sampleCnt);

        /* Copy Q samples to space after I samples space */
        pMsg->connlessIQReport.pQSample = (int8_t *)((uint8_t *) pMsg->connlessIQReport.pISample + pDmEvt->connlessIQReport.sampleCnt);
        memcpy(pMsg->connlessIQReport.pQSample, pDmEvt->connlessIQReport.pQSample, pDmEvt->connlessIQReport.sampleCnt);
      }

      WsfMsgSend(centralPastCb.handlerId, pMsg);
    }
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Application  ATT callback.
 *
 *  \param  pEvt    ATT callback event
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralPastAttCback(attEvt_t *pEvt)
{
  attEvt_t *pMsg;

  if ((pMsg = WsfMsgAlloc(sizeof(attEvt_t) + pEvt->valueLen)) != NULL)
  {
    memcpy((uint8_t *)pMsg, (uint8_t *)pEvt, sizeof(attEvt_t));
    pMsg->pValue = (uint8_t *) (pMsg + 1);
    memcpy(pMsg->pValue, pEvt->pValue, pEvt->valueLen);
    WsfMsgSend(centralPastCb.handlerId, pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Perform actions on scan start.
 *
 *  \param  pMsg    Pointer to DM callback event message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralPastScanStart(dmEvt_t *pMsg)
{
  if (pMsg->hdr.status == HCI_SUCCESS)
  {
    centralPastCb.scanning = TRUE;
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Perform actions on scan stop.
 *
 *  \param  pMsg    Pointer to DM callback event message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralPastScanStop(dmEvt_t *pMsg)
{
  if (pMsg->hdr.status == HCI_SUCCESS)
  {
    centralPastCb.scanning = FALSE;
    centralPastCb.autoConnect = FALSE;

    /* Open connection */
    if (centralPastConnInfo.doConnect)
    {
      if ( (centralPastConnInfo.eventType & HCI_ADV_RPT_LEG_ADV_BIT) == 0 )
      {
        APP_TRACE_INFO1("AUX Connect, Secondary Advertising PHY = %d", centralPastConnInfo.secPhy);
        AppExtConnOpen(HCI_INIT_PHY_LE_1M_BIT | centralPastConnInfo.secPhy, centralPastConnInfo.addrType, centralPastConnInfo.addr, centralPastConnInfo.dbHdl);
      }
      else
      {
        APP_TRACE_INFO0("Legacy Connect");
        AppExtConnOpen(HCI_INIT_PHY_LE_1M_BIT, centralPastConnInfo.addrType, centralPastConnInfo.addr, centralPastConnInfo.dbHdl);
      }

      centralPastConnInfo.doConnect = FALSE;
    }
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Handle a extended scan report.
 *
 *  \param  pMsg    Pointer to DM callback event message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralPastExtScanReport(dmEvt_t *pMsg)
{
  uint8_t    *pData;
  appDbHdl_t dbHdl;
  bool_t     connect = FALSE;
  char tmp_name[EXT_SCAN_NAME_MAX_LEN + 1] = {0};

  /* disregard if not scanning */
  if (!centralPastCb.scanning)
  {
    return;
  }

  /* if we already have a bond with this device then connect to it */
  if ((dbHdl = AppDbFindByAddr(pMsg->extScanReport.addrType, pMsg->extScanReport.addr)) != APP_DB_HDL_NONE)
  {
    /* if this is a directed advertisement where the initiator address is an RPA */
    if (DM_RAND_ADDR_RPA(pMsg->extScanReport.directAddr, pMsg->extScanReport.directAddrType))
    {
      /* resolve direct address to see if it's addressed to us */
      AppMasterResolveAddr(pMsg, dbHdl, APP_RESOLVE_DIRECT_RPA);
    }
    else
    {
      APP_TRACE_INFO0("Connecting to a bonded device\n");
      connect = TRUE;
    }
  }
  /* if the peer device uses an RPA */
  else if (DM_RAND_ADDR_RPA(pMsg->extScanReport.addr, pMsg->extScanReport.addrType))
  {
    /* resolve advertiser's RPA to see if we already have a bond with this device */
    AppMasterResolveAddr(pMsg, APP_DB_HDL_NONE, APP_RESOLVE_ADV_RPA);
  }
  /* find vendor-specific advertising data */
  else if ((pData = DmFindAdType(DM_ADV_TYPE_MANUFACTURER, pMsg->extScanReport.len,
                                 pMsg->extScanReport.pData)) != NULL)
  {
    /* check length and vendor ID */
    if (pData[DM_AD_LEN_IDX] >= 3 && BYTES_UINT16_CMP(&pData[DM_AD_DATA_IDX], HCI_ID_PACKETCRAFT))
    {
      //connect = TRUE;
    }
  }
  /* find Local name advertising data */
  else if ((pData = DmFindAdType(DM_ADV_TYPE_LOCAL_NAME, pMsg->extScanReport.len,
                                 pMsg->extScanReport.pData)) != NULL)
  {
    memcpy(tmp_name, (char *)(pData + 2), *pData - 1);

    APP_TRACE_INFO1("Scanned device name: %s", tmp_name);

    /* check length and vendor ID */
    if ( !strncmp((char *)(pData + 2), "Am_Per_Adv", *pData - 1) )
    {
      centralPastConnInfo.advSid = pMsg->extScanReport.advSid;
      centralPastConnInfo.advAddrType = pMsg->extScanReport.addrType;
      memcpy(centralPastConnInfo.advAddr, pMsg->extScanReport.addr, BDA_ADDR_LEN);

      APP_TRACE_INFO0("Found expected periodic advertising device");

#if (BT_53)
#define INIT_PER_ADV_FILT
#if defined(INIT_PER_ADV_FILT)
      DmSyncInitialDupFiltEnable(1);
#endif // defined(INIT_PER_ADV_FILT)

    if ( (AppConnIsOpen() == DM_CONN_ID_NONE) || (centralPastConnInfo.synced) )
    {
      return;
    }
    else
    {
      connect = FALSE;
    }
#endif // BT_53

      uint16_t skip = 0;             // Do not skip any periodic advertising packets
      uint16_t syncTimeout = 0x03E8; // 1000 * 10ms
      AppSyncStart(centralPastConnInfo.advSid, centralPastConnInfo.advAddrType,
                   centralPastConnInfo.advAddr, skip, syncTimeout);
      am_util_stdio_printf("Synchronizing with periodic advertising from device: " "%02X:%02X:%02X:%02X:%02X:%02X\n",
                           centralPastConnInfo.advAddr[5], centralPastConnInfo.advAddr[4],
                           centralPastConnInfo.advAddr[3], centralPastConnInfo.advAddr[2],
                           centralPastConnInfo.advAddr[1], centralPastConnInfo.advAddr[0]);
      centralPastConnInfo.synced = TRUE;
    }
#if (BT_53)
#if defined(BLE_PAST)
    else if ((!strncmp((char *)(pData + 2), "Ambq_AE", *pData - 1))
        || (!strncmp((char *)(pData + 2), "Ambq_AE2", *pData - 1)))
    {
      bdAddr_t advAddr;

      memcpy(advAddr, pMsg->extScanReport.addr, BDA_ADDR_LEN);

      APP_TRACE_INFO0("Found expected advertising device");
      centralPastConnInfo.addrType = DmHostAddrType(pMsg->extScanReport.addrType);
      memcpy(centralPastConnInfo.addr, pMsg->extScanReport.addr, sizeof(bdAddr_t));

      AppExtScanStop();
    }
#endif // defined(BLE_PAST)
#endif // BT_53
  }

  if (connect)
  {
    /* stop scanning and connect */
    centralPastCb.autoConnect = FALSE;
    AppExtScanStop();

    /* Store peer information for connect on scan stop */
    centralPastConnInfo.addrType = DmHostAddrType(pMsg->extScanReport.addrType);
    memcpy(centralPastConnInfo.addr, pMsg->extScanReport.addr, sizeof(bdAddr_t));
    centralPastConnInfo.dbHdl = dbHdl;
    centralPastConnInfo.doConnect = TRUE;
    centralPastConnInfo.secPhy = pMsg->extScanReport.secPhy;
    centralPastConnInfo.eventType = pMsg->extScanReport.eventType;
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Perform UI actions on connection open.
 *
 *  \param  pMsg    Pointer to DM callback event message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralPastOpen(dmEvt_t *pMsg)
{
  /* Set the antenna identifiers for the connection */
  AtpcSetAntennaIds((dmConnId_t) pMsg->hdr.param, sizeof(centralPastAntennaIds), centralPastAntennaIds);

  /* Reset control information. */
  centralPastCb.cteAction[pMsg->hdr.param-1] = CENTRAL_PAST_CTE_ACTION_ENABLE;

#if (BT_53)
#if defined(BLE_PAST)
  centralPastConnInfo.conn_id = (dmConnId_t) pMsg->hdr.param;
#endif // defined(BLE_PAST)
#endif // BT_53
}

/*************************************************************************************************/
/*!
 *  \brief  Process a received ATT notification.
 *
 *  \param  pMsg    Pointer to ATT callback event message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralPastValueNtf(attEvt_t *pMsg)
{
}

/*************************************************************************************************/
/*!
 *  \brief  Process IQ report from the event handler.
 *
 *  \param  pIqRpt  Pointer to IQ Report message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralPastProcIqRpt(hciLeConnIQReportEvt_t *pIqRpt)
{
  APP_TRACE_INFO0("IQ Report");
  APP_TRACE_INFO1("  status: %x", pIqRpt->hdr.status);
  APP_TRACE_INFO1("  sampleCnt: %d", pIqRpt->sampleCnt);
}


/*************************************************************************************************/
/*!
 *  \brief  Process connectionless IQ report from the event handler.
 *
 *  \param  pIqRpt  Pointer to IQ Report message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralPastProcConlessIqRpt(hciLeConlessIQReportEvt_t *pIqRpt)
{
  APP_TRACE_INFO0("connectionless IQ Report");
  APP_TRACE_INFO1("  status: %x", pIqRpt->hdr.status);
  APP_TRACE_INFO1("  sampleCnt: %d", pIqRpt->sampleCnt);
}

/*************************************************************************************************/
/*!
 *  \brief  Set up procedures that need to be performed after device reset.
 *
 *  \param  pMsg    Pointer to DM callback event message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralPastSetup(dmEvt_t *pMsg)
{
  centralPastCb.scanning = FALSE;
  centralPastCb.autoConnect = FALSE;
  centralPastConnInfo.doConnect = FALSE;

  DmConnSetConnSpec((hciConnSpec_t *) &centralPastConnCfg);
}

/*************************************************************************************************/
/*!
 *  \brief  GAP service discovery has completed.
 *
 *  \param  connId    Connection identifier.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralPastDiscGapCmpl(dmConnId_t connId)
{
  appDbHdl_t dbHdl;

  /* if RPA Only attribute found on peer device */
  if ((pCentralPastGapHdlList[connId-1][GAP_RPAO_HDL_IDX] != ATT_HANDLE_NONE) &&
      ((dbHdl = AppDbGetHdl(connId)) != APP_DB_HDL_NONE))
  {
    /* update DB */
    AppDbSetPeerRpao(dbHdl, TRUE);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Button press callback.
 *
 *  \param  btn    Button press.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralPastBtnCback(uint8_t btn)
{
  dmConnId_t  connId = centralPastCb.btnConnId;
  dmConnId_t  connIdList[DM_CONN_MAX];
  uint8_t     numConnections = AppConnOpenList(connIdList);

  /* button actions when connected */
  if (numConnections > 0)
  {
    switch (btn)
    {
      case APP_UI_BTN_1_SHORT:
        if (numConnections < DM_CONN_MAX - 1)
        {
          /* if scanning cancel scanning */
          if (centralPastCb.scanning)
          {
            AppExtScanStop();
          }
          /* else auto connect */
          else if (!centralPastCb.autoConnect)
          {
            centralPastCb.autoConnect = TRUE;
            centralPastConnInfo.doConnect = FALSE;
            AppExtScanStart(HCI_SCAN_PHY_LE_1M_BIT, centralPastMasterExtCfg.discMode,
                centralPastMasterExtCfg.scanType, centralPastMasterExtCfg.scanDuration, centralPastMasterExtCfg.scanPeriod);
          }
        }
        else
        {
          APP_TRACE_INFO0("centralPastBtnCback: Max connections reached.");
        }
        break;

      case APP_UI_BTN_1_MED:
        /* Increment which connection ID is used in button presses */
        if (++centralPastCb.btnConnId > DM_CONN_MAX)
        {
          centralPastCb.btnConnId = 1;
        }
        APP_TRACE_INFO1("ConnId for Button Press: %d", centralPastCb.btnConnId);
        break;

      case APP_UI_BTN_1_LONG:
        /* disconnect */
        AppConnClose(connId);
        break;

      case APP_UI_BTN_1_EX_LONG:
        if (centralPastCb.cteAction[connId-1] == CENTRAL_PAST_CTE_ACTION_ENABLE)
        {
          /* Toggle action. */
          centralPastCb.cteAction[connId-1] = CENTRAL_PAST_CTE_ACTION_DISABLE;

          /* Perform AoA enable request. */
          AtpcCteAclEnableReq(connId, pCentralPastCteHdlList[connId-1][ATPC_CTE_ENABLE_HDL],
                              CENTRAL_PAST_CTE_MIN_LEN, CENTRAL_PAST_CTE_INTERVAL, CENTRAL_PAST_CTE_TYPE);
        }
        else
        {
          /* Toggle action. */
          centralPastCb.cteAction[connId-1] = CENTRAL_PAST_CTE_ACTION_ENABLE;

          /* Perform AoA disable request. */
          //AtpcCteAclDisableReq(connId, pCentralPastCteHdlList[connId-1][ATPC_CTE_ENABLE_HDL]);
          DmConnCteReqStop(connId);
        }
        break;

      default:
        break;
    }
  }
  /* button actions when not connected */
  else
  {
    switch (btn)
    {
      case APP_UI_BTN_1_SHORT:
        /* if scanning cancel scanning */
        if (centralPastCb.scanning)
        {
          AppExtScanStop();
        }
        /* else auto connect */
        else if (!centralPastCb.autoConnect)
        {
          centralPastCb.autoConnect = TRUE;
          centralPastConnInfo.doConnect = FALSE;
          DmSyncInitialRptEnable(TRUE);
          AppExtScanStart(HCI_SCAN_PHY_LE_1M_BIT, centralPastMasterExtCfg.discMode,
                centralPastMasterExtCfg.scanType, centralPastMasterExtCfg.scanDuration, centralPastMasterExtCfg.scanPeriod);
        }
        break;

      case APP_UI_BTN_1_MED:
        /* Increment which connection ID is used in button presses */
        if (++centralPastCb.btnConnId > DM_CONN_MAX)
        {
          centralPastCb.btnConnId = 1;
        }
        APP_TRACE_INFO1("ConnID for Button Press: %d", centralPastCb.btnConnId);
        break;

      case APP_UI_BTN_1_LONG:
        /* clear all bonding info */
        AppClearAllBondingInfo();
        break;

      case APP_UI_BTN_1_EX_LONG:
        /* add RPAO characteristic to GAP service -- needed only when DM Privacy enabled */
        SvcCoreGapAddRpaoCh();
        break;

      case APP_UI_BTN_2_EX_LONG:
        /* enable device privacy -- start generating local RPAs every 15 minutes */
        DmDevPrivStart(15 * 60);

        /* set Scanning filter policy to accept directed advertisements with RPAs */
        DmDevSetFilterPolicy(DM_FILT_POLICY_MODE_SCAN, HCI_FILT_RES_INIT);
        break;

      default:
        break;
    }
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Discovery callback.
 *
 *  \param  connId    Connection identifier.
 *  \param  status    Service or configuration status.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralPastDiscCback(dmConnId_t connId, uint8_t status)
{
  switch(status)
  {
    case APP_DISC_INIT:
      /* set handle list when initialization requested */
      AppDiscSetHdlList(connId, centralPastCb.hdlListLen, centralPastCb.hdlList[connId-1]);
      break;

    case APP_DISC_READ_DATABASE_HASH:
      /* read peer's database hash */
      AppDiscReadDatabaseHash(connId);
      break;

    case APP_DISC_SEC_REQUIRED:
      /* initiate security */
      AppMasterSecurityReq(connId);
      break;

    case APP_DISC_START:
      /* initialize discovery state */
      centralPastCb.discState[connId-1] = CENTRAL_PAST_DISC_GATT_SVC;

      /* discover GATT service */
      GattDiscover(connId, pCentralPastGattHdlList[connId-1]);
      break;

    case APP_DISC_FAILED:
      if (pAppCfg->abortDisc)
      {
        /* if discovery failed for proprietary data service then disconnect */
        if (centralPastCb.discState[connId-1] == CENTRAL_PAST_DISC_CTE_SVC)
        {
          AppConnClose(connId);
          break;
        }
      }
      /* Else falls through. */

    case APP_DISC_CMPL:
      /* next discovery state */
      centralPastCb.discState[connId-1]++;

      if (centralPastCb.discState[connId-1] == CENTRAL_PAST_DISC_GAP_SVC)
      {
        /* discover GAP service */
        GapDiscover(connId, pCentralPastGapHdlList[connId-1]);
      }
      else if (centralPastCb.discState[connId-1] == CENTRAL_PAST_DISC_CTE_SVC)
      {
        /* discover proprietary data service */
        AtpcCteDiscover(connId, pCentralPastCteHdlList[connId-1]);
      }
      else
      {
        /* discovery complete */
        AppDiscComplete(connId, APP_DISC_CMPL);

        /* GAP service discovery completed */
        centralPastDiscGapCmpl(connId);

        /* start configuration */
        AppDiscConfigure(connId, APP_DISC_CFG_START, CENTRAL_PAST_DISC_CFG_LIST_LEN,
                         (attcDiscCfg_t *) centralPastDiscCfgList, CENTRAL_PAST_DISC_HDL_LIST_LEN, centralPastCb.hdlList[connId-1]);
      }
      break;

    case APP_DISC_CFG_START:
        /* start configuration */
        AppDiscConfigure(connId, APP_DISC_CFG_START, CENTRAL_PAST_DISC_CFG_LIST_LEN,
                         (attcDiscCfg_t *) centralPastDiscCfgList, CENTRAL_PAST_DISC_HDL_LIST_LEN, centralPastCb.hdlList[connId-1]);
      break;

    case APP_DISC_CFG_CMPL:
      AppDiscComplete(connId, status);
#if (BT_53)
#if defined(BLE_PAST)
      centralPastCb.autoConnect = FALSE;
      if (centralPastConnInfo.synced)
      {
        /* Send synchronization information to the connected device if already synced */
        uint16_t service_data = 0x0000;
        DmPastSyncTrsf(centralPastConnInfo.conn_id, service_data, centralPastConnInfo.syncId);
      }
      else
      {
        /* Start scanning if no periodic advertising device synced*/
        AppExtScanStart(HCI_SCAN_PHY_LE_1M_BIT, centralPastMasterExtCfg.discMode,
                  centralPastMasterExtCfg.scanType, 0, centralPastMasterExtCfg.scanPeriod);
      }
#endif // defined(BLE_PAST)
#endif // BT_53
      break;

    case APP_DISC_CFG_CONN_START:
      /* no connection setup configuration */
      break;

    default:
      break;
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Process messages from the event handler.
 *
 *  \param  pMsg    Pointer to message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralPastProcMsg(dmEvt_t *pMsg)
{
  uint8_t uiEvent = APP_UI_NONE;

  switch(pMsg->hdr.event)
  {
    case ATTC_HANDLE_VALUE_NTF:
      centralPastValueNtf((attEvt_t *) pMsg);
      break;

    case DM_RESET_CMPL_IND:
      AttsCalculateDbHash();
      DmSecGenerateEccKeyReq();
      centralPastSetup(pMsg);
      centralPastBtnCback(APP_UI_BTN_1_SHORT);
      uiEvent = APP_UI_RESET_CMPL;
      break;

    case DM_EXT_SCAN_START_IND:
      centralPastScanStart(pMsg);
      uiEvent = APP_UI_SCAN_START;
      break;

    case DM_EXT_SCAN_STOP_IND:
      centralPastScanStop(pMsg);
      uiEvent = APP_UI_SCAN_STOP;
#if (BT_53)
#if defined(BLE_PAST)
      AppExtConnOpen(HCI_PHY_LE_1M_BIT, centralPastConnInfo.addrType, centralPastConnInfo.addr, NULL);
#endif // defined(BLE_PAST)
#endif // BT_53
      break;

    case DM_EXT_SCAN_REPORT_IND:
      centralPastExtScanReport(pMsg);
      break;

      case DM_PER_ADV_REPORT_IND:
      {
        uint16_t syncHandle = pMsg->perAdvReport.syncHandle;
        uint8_t slotDurations = 0x01;
        uint8_t maxSampleCte = 0x0;
        uint8_t switchPatternLen = 0x2;
        uint8_t pAntennaIDs[HCI_MIN_NUM_ANTENNA_IDS] = {0, 1};

        HciLeSetConlessIQSampleEnCmd(syncHandle, TRUE, slotDurations,
                                     maxSampleCte, switchPatternLen, pAntennaIDs);

        break;
      }

      case DM_CONNLESS_IQ_REPORT_IND:
      {
        centralPastProcConlessIqRpt(&pMsg->connlessIQReport);

        break;
      }

    case DM_CONN_OPEN_IND:
      centralPastOpen(pMsg);
      uiEvent = APP_UI_CONN_OPEN;
      break;

    case DM_CONN_CLOSE_IND:
      AppExtScanStart(HCI_SCAN_PHY_LE_1M_BIT, centralPastMasterExtCfg.discMode,
            centralPastMasterExtCfg.scanType, centralPastMasterExtCfg.scanDuration, centralPastMasterExtCfg.scanPeriod);
      uiEvent = APP_UI_CONN_CLOSE;
      break;

    case DM_PHY_UPDATE_IND:
      APP_TRACE_INFO3("DM_PHY_UPDATE_IND status: %d, RX: %d, TX: %d", pMsg->phyUpdate.status, pMsg->phyUpdate.rxPhy, pMsg->phyUpdate.txPhy);
      break;

    case DM_SEC_PAIR_CMPL_IND:
      DmSecGenerateEccKeyReq();
      uiEvent = APP_UI_SEC_PAIR_CMPL;
      break;

    case DM_SEC_PAIR_FAIL_IND:
      DmSecGenerateEccKeyReq();
      uiEvent = APP_UI_SEC_PAIR_FAIL;
      break;

    case DM_SEC_ENCRYPT_IND:
      uiEvent = APP_UI_SEC_ENCRYPT;
      break;

    case DM_SEC_ENCRYPT_FAIL_IND:
      uiEvent = APP_UI_SEC_ENCRYPT_FAIL;
      break;

    case DM_SEC_AUTH_REQ_IND:

      if (pMsg->authReq.oob)
      {
        dmConnId_t connId = (dmConnId_t) pMsg->hdr.param;
        if (centralPastOobCfg != NULL)
        {
          DmSecSetOob(connId, centralPastOobCfg);
        }

        DmSecAuthRsp(connId, 0, NULL);
      }
      else
      {
        AppHandlePasskey(&pMsg->authReq);
      }
      break;

    case DM_SEC_COMPARE_IND:
      AppHandleNumericComparison(&pMsg->cnfInd);
      break;

    case DM_ADV_NEW_ADDR_IND:
      break;

    case DM_PRIV_CLEAR_RES_LIST_IND:
      APP_TRACE_INFO1("Clear resolving list status 0x%02x", pMsg->hdr.status);
      break;

    case DM_CONN_IQ_REPORT_IND:
      centralPastProcIqRpt(&pMsg->connIQReport);
      break;

    case DM_CTE_REQ_FAIL_IND:
      APP_TRACE_INFO1("DM_CTE_REQ_FAIL_IND: status: %#x", pMsg->hdr.status);
      break;

    default:
      break;
  }

  if (uiEvent != APP_UI_NONE)
  {
    AppUiAction(uiEvent);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Application handler init function called during system initialization.
 *
 *  \param  handlerID  WSF handler ID.
 *
 *  \return None.
 */
/*************************************************************************************************/
void CentralPastHandlerInit(wsfHandlerId_t handlerId)
{
  /* store handler ID */
  centralPastCb.handlerId = handlerId;

  /* set handle list length */
  centralPastCb.hdlListLen = CENTRAL_PAST_DISC_HDL_LIST_LEN;

  centralPastCb.btnConnId = 1;
  /* Set configuration pointers */
  pAppExtMasterCfg = (appExtMasterCfg_t *)&centralPastMasterExtCfg;
  pAppSecCfg = (appSecCfg_t *) &centralPastSecCfg;
  pAppDiscCfg = (appDiscCfg_t *) &centralPastDiscCfg;
  pAppCfg = (appCfg_t *)&centralPastAppCfg;
  pSmpCfg = (smpCfg_t *) &centralPastSmpCfg;
  pAttCfg = (attCfg_t *) &centralPastAttCfg;

  /* Initialize application framework */
  AppMasterInit();
  AppDiscInit();

  /* Set IRK for the local device */
  DmSecSetLocalIrk(localIrk);
}

/*************************************************************************************************/
/*!
 *  \brief  WSF event handler for application.
 *
 *  \param  event   WSF event mask.
 *  \param  pMsg    WSF message.
 *
 *  \return None.
 */
/*************************************************************************************************/
void CentralPastHandler(wsfEventMask_t event, wsfMsgHdr_t *pMsg)
{
  if (pMsg != NULL)
  {
    /* process ATT messages */
    if (pMsg->event <= ATT_CBACK_END)
    {
      /* process discovery-related ATT messages */
      AppDiscProcAttMsg((attEvt_t *) pMsg);

      /* process server-related ATT messages */
      AppServerProcAttMsg(pMsg);
    }
    /* process DM messages */
    else if (pMsg->event <= DM_CBACK_END)
    {
      /* process advertising and connection-related messages */
      AppMasterProcDmMsg((dmEvt_t *) pMsg);

      /* process security-related messages */
      AppMasterSecProcDmMsg((dmEvt_t *) pMsg);

      /* process discovery-related messages */
      AppDiscProcDmMsg((dmEvt_t *) pMsg);
    }

    /* process asset tracking profile related messages */
    AtpcProcMsg(pMsg);

    /* perform profile and user interface-related operations */
    centralPastProcMsg((dmEvt_t *) pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Initialize the pointers into the handle list.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralPastInitSvcHdlList(void)
{
  uint8_t i;

  for (i = 0; i < DM_CONN_MAX; i++)
  {
    /*! Pointers into handle list for each service's handles */
    pCentralPastGattHdlList[i] = &centralPastCb.hdlList[i][CENTRAL_PAST_DISC_GATT_START];
    pCentralPastGapHdlList[i] = &centralPastCb.hdlList[i][CENTRAL_PAST_DISC_GAP_START];
    pCentralPastCteHdlList[i] = &centralPastCb.hdlList[i][CENTRAL_PAST_DISC_CTE_START];
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Start the application.
 *
 *  \return None.
 */
/*************************************************************************************************/
void CentralPastStart(void)
{
  /* Initialize handle pointers */
  centralPastInitSvcHdlList();

  /* Register for stack callbacks */
  DmRegister(centralPastDmCback);
  DmConnRegister(DM_CLIENT_ID_APP, centralPastDmCback);
  AttRegister(centralPastAttCback);

  /* Register for app framework button callbacks */
  AppUiBtnRegister(centralPastBtnCback);

  /* Register for app framework discovery callbacks */
  AppDiscRegister(centralPastDiscCback);

  /* Initialize attribute server database */
  SvcCoreAddGroup();

  /* Reset the device */
  DmDevReset();
}
