// ****************************************************************************
//
//  central_subrate_main.c
//! @file
//!
//! @brief Central subrate sample application.
//!
//! @{
//
// ****************************************************************************

//*****************************************************************************
//
// Copyright (c) 2026, Ambiq Micro, Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the copyright holder nor the names of its
// contributors may be used to endorse or promote products derived from this
// software without specific prior written permission.
//
// Third party software included in this distribution is subject to the
// additional license terms as defined in the /docs/licenses directory.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// This is part of revision stable-e24d618f43 of the AmbiqSuite Development Package.
//
//*****************************************************************************

#include <string.h>
#include "wsf_types.h"
#include "util/bstream.h"
#include "wsf_msg.h"
#include "wsf_trace.h"
#include "wsf_assert.h"
#include "hci_api.h"
#include "dm_api.h"
#include "att_api.h"
#include "smp_api.h"
#include "app_cfg.h"
#include "app_api.h"
#include "app_db.h"
#include "app_ui.h"
#include "app_main.h"
#include "svc_core.h"
#include "svc_ch.h"
#include "gatt/gatt_api.h"
#include "tipc/tipc_api.h"
#include "anpc/anpc_api.h"
#include "paspc/paspc_api.h"
#include "hrpc/hrpc_api.h"
#include "dis/dis_api.h"
#include "atts_main.h"

#ifdef AM_BLE_EATT
#include "att_eatt.h"
#include "eatt_api.h"
#endif

/**************************************************************************************************
  Local Variables
**************************************************************************************************/
typedef enum
{
    CENTRAL_SUBRATE_SCAN_NONE,
    CENTRAL_SUBRATE_SCAN_START,
    CENTRAL_SUBRATE_SCANNING
}eCentralSubrateScanState;
/*! application control block */
static struct
{
  uint16_t          hdlMasterList[APP_DB_HDL_LIST_LEN];   /*! Cached handle list in master role */
  uint16_t          hdlSlaveList[APP_DB_HDL_LIST_LEN];    /*! Cached handle list in slave role */
  wsfHandlerId_t    handlerId;                            /*! WSF hander ID */
  uint8_t           discState;                            /*! Service discovery state */
  eCentralSubrateScanState   scan_state;
  bool_t            autoConnect;                          /*! TRUE if auto-connecting */
} centralSubrateCb;

/*! \brief application connection information */
typedef struct
{
  appDbHdl_t        dbHdl;                                /*! Device database record handle type */
  uint8_t           addrType;                             /*! Type of address of device to connect to */
  bdAddr_t          addr;                                 /*! Address of device to connect to */
  bool_t            doConnect;                            /*! TRUE to issue connect on scan complete */
  int8_t            rssi;                                 /*! RSSI of discovered remote device  */
} centralSubrateConnInfo_t;

centralSubrateConnInfo_t centralSubrateConnInfo;

/**************************************************************************************************
  Configurable Parameters
**************************************************************************************************/

/*! configurable parameters for security */
static const appSecCfg_t centralSubrateSecCfg =
{
  DM_AUTH_BOND_FLAG,                      /*! Authentication and bonding flags */
  0,                                      /*! Initiator key distribution flags */
  DM_KEY_DIST_LTK,                        /*! Responder key distribution flags */
  FALSE,                                  /*! TRUE if Out-of-band pairing data is present */
  TRUE                                    /*! TRUE to initiate security upon connection */
};

/*! configurable parameters for connection parameter update */
static const appUpdateCfg_t centralSubrateUpdateCfg =
{
  0,                                      /*! Connection idle period in ms before attempting
                                              connection parameter update; set to zero to disable */
  600,                                    /*! Minimum connection interval in 1.25ms units */
  800,                                    /*! Maximum connection interval in 1.25ms units */
  3,                                      /*! Connection latency */
  600,                                    /*! Supervision timeout in 10ms units */
  5                                       /*! Number of update attempts before giving up */
};

/*! Connection parameters */
static const hciConnSpec_t centralSubrateConnCfg =
{
  160,                                    /*! Minimum connection interval in 1.25ms units */
  160,                                    /*! Maximum connection interval in 1.25ms units */
  0,                                      /*! Connection latency */
  600,                                    /*! Supervision timeout in 10ms units */
  0,                                      /*! Unused */
  0                                       /*! Unused */
};

/*! SMP security parameter configuration */
/* Configuration structure */
static const smpCfg_t centralSubrateSmpCfg =
{
  3000,                                   /*! 'Repeated attempts' timeout in msec */
  SMP_IO_NO_IN_NO_OUT,                    /*! I/O Capability */
  7,                                      /*! Minimum encryption key length */
  16,                                     /*! Maximum encryption key length */
  3,                                      /*! Attempts to trigger 'repeated attempts' timeout */
  0,                                      /*! Device authentication requirements */
  64000,                                  /*! Maximum repeated attempts timeout in msec */
  64000,                                  /*! Time msec before attemptExp decreases */
  2                                       /*! Repeated attempts multiplier exponent */
};

#ifdef AM_BLE_EATT
/* EATT Configuration structure */
static const eattCfg_t centralSubrateeattCfg =
{
  247,                              /* MTU */
  247,                              /* MPS */
  TRUE,                             /* Open EATT channels automatically on connect */
  FALSE,                            /* Authorization required */
  DM_SEC_LEVEL_NONE,                /* Security level required */
  2,                                /* Number of enhanced l2cap channels per connection */
  NULL                              /* Channel priority table */
};
#endif

/*! Configurable parameters for service and characteristic discovery */
static const appDiscCfg_t centralSubrateDiscCfg =
{
  FALSE,                                  /*! TRUE to wait for a secure connection before initiating discovery */
  FALSE                                   /*! TRUE to fall back on database hash to verify handles when no bond exists. */
};

static const appCfg_t centralSubrateAppCfg =
{
  TRUE,                                   /*! TRUE to abort service discovery if service not found */
  TRUE                                    /*! TRUE to disconnect if ATT transaction times out */
};

/*! configurable parameters for master */
static const appMasterCfg_t centralSubrateMasterCfg =
{
  96,                                      /*! The scan interval, in 0.625 ms units */
  48,                                      /*! The scan window, in 0.625 ms units  */
  4000,                                    /*! The scan duration in ms */
  DM_DISC_MODE_NONE,                       /*! The GAP discovery mode */
  DM_SCAN_TYPE_ACTIVE                      /*! The scan type (active or passive) */
};

static const appSubrateCfg_t centralSubrateDefaultSubrateCfg =
{
  1,                                       /*! Minimum subrate factor */
  6,                                       /*! Maximum subrate factor */
  1,                                       /*! Maximum peripheral latency */
  0,                                       /*! Continuation number */
  600                                      /*! Supervision timeout in 10ms units */
};

/**************************************************************************************************
  ATT Client Discovery Data
**************************************************************************************************/

/*! Discovery states:  enumeration of services to be discovered */
enum
{
  CENTRAL_SUBRATE_DISC_SLAVE_GATT_SVC,      /* GATT service */
  CENTRAL_SUBRATE_DISC_SLAVE_CTS_SVC,       /* Current Time service */
  CENTRAL_SUBRATE_DISC_SLAVE_ANS_SVC,       /* Alert Notification service */
  CENTRAL_SUBRATE_DISC_SLAVE_PASS_SVC,      /* Phone Alert Status service */
  CENTRAL_SUBRATE_DISC_SLAVE_SVC_MAX        /* Discovery complete */
};

/*! Discovery states:  enumeration of services to be discovered in the master role */
enum
{
  CENTRAL_SUBRATE_DISC_MASTER_GATT_SVC,      /* GATT service */
  CENTRAL_SUBRATE_DISC_MASTER_DIS_SVC,       /* Device Information service */
  CENTRAL_SUBRATE_DISC_MASTER_HRS_SVC,       /* Heart Rate service */
};

/*! the Client handle list, centralSubrateCb.hdlList[], is set as follows:
 *
 *  ------------------------------- <- CENTRAL_SUBRATE_DISC_GATT_START
 *  | GATT handles                |
 *  |...                          |
 *  ------------------------------- <- CENTRAL_SUBRATE_DISC_CTS_START
 *  | CTS handles                 |
 *  | ...                         |
 *  ------------------------------- <- CENTRAL_SUBRATE_DISC_ANS_START
 *  | ANS handles                 |
 *  | ...                         |
 *  ------------------------------- <- CENTRAL_SUBRATE_DISC_PASS_START
 *  | PASS handles                |
 *  | ...                         |
 *  -------------------------------
 */

/*! Start of each service's handles in the the handle list */
#define CENTRAL_SUBRATE_DISC_GATT_START           0
#define CENTRAL_SUBRATE_DISC_CTS_START            (CENTRAL_SUBRATE_DISC_GATT_START + GATT_HDL_LIST_LEN)
#define CENTRAL_SUBRATE_DISC_ANS_START            (CENTRAL_SUBRATE_DISC_CTS_START + TIPC_CTS_HDL_LIST_LEN)
#define CENTRAL_SUBRATE_DISC_PASS_START           (CENTRAL_SUBRATE_DISC_ANS_START + ANPC_ANS_HDL_LIST_LEN)
#define CENTRAL_SUBRATE_DISC_SLAVE_HDL_LIST_LEN   (CENTRAL_SUBRATE_DISC_PASS_START + PASPC_PASS_HDL_LIST_LEN)

/* Start of cached heart rate service handles; begins after GATT - Master Role */
#define CENTRAL_SUBRATE_DISC_DIS_START            (CENTRAL_SUBRATE_DISC_GATT_START + GATT_HDL_LIST_LEN)
#define CENTRAL_SUBRATE_DISC_HRS_START            (CENTRAL_SUBRATE_DISC_DIS_START + DIS_HDL_LIST_LEN)
#define CENTRAL_SUBRATE_DISC_MASTER_HDL_LIST_LEN  (CENTRAL_SUBRATE_DISC_HRS_START + HRPC_HRS_HDL_LIST_LEN)

/*! Pointers into handle list heart rate service handles - Master Role */
static uint16_t *pCentralSubrateMstGattHdlList = &centralSubrateCb.hdlMasterList[CENTRAL_SUBRATE_DISC_GATT_START];
static uint16_t *pCentralSubrateDisHdlList =     &centralSubrateCb.hdlMasterList[CENTRAL_SUBRATE_DISC_DIS_START];
static uint16_t *pCentralSubrateHrsHdlList =     &centralSubrateCb.hdlMasterList[CENTRAL_SUBRATE_DISC_HRS_START];

/* sanity check:  make sure handle list length is <= app db handle list length */
WSF_CT_ASSERT(CENTRAL_SUBRATE_DISC_SLAVE_HDL_LIST_LEN <= APP_DB_HDL_LIST_LEN);
WSF_CT_ASSERT(CENTRAL_SUBRATE_DISC_MASTER_HDL_LIST_LEN <= APP_DB_HDL_LIST_LEN);

/**************************************************************************************************
  ATT Client Configuration Data
**************************************************************************************************/

/*
 * Data for configuration after service discovery
 */

/* Default value for CCC notifications */
static const uint8_t centralSubrateCccNtfVal[] = {UINT16_TO_BYTES(ATT_CLIENT_CFG_NOTIFY)};

/* Default value for Client Supported Features (enable Robust Caching and Enhanced ATT Bearer) */
static const uint8_t centralSubrateCsfVal[1] = { ATTS_CSF_ROBUST_CACHING | ATTS_CSF_EATT_BEARER };

/* HRS Control point "Reset Energy Expended" */
static const uint8_t centralSubrateHrsRstEnExp[] = {CH_HRCP_RESET_ENERGY_EXP};

/* List of characteristics to configure after service discovery - Master Role */
static const attcDiscCfg_t centralSubrateDiscMasterCfgList[] =
{
  /* Write:  GATT client supported features */
  { centralSubrateCsfVal, sizeof(centralSubrateCsfVal), (GATT_CSF_HDL_IDX + CENTRAL_SUBRATE_DISC_GATT_START) },

  /* Read:  DIS Manufacturer name string */
  {NULL, 0, DIS_MFNS_HDL_IDX},

  /* Read:  DIS Model number string */
  {NULL, 0, DIS_MNS_HDL_IDX},

  /* Read:  DIS Serial number string */
  {NULL, 0, DIS_SNS_HDL_IDX},

  /* Read:  DIS Hardware revision string */
  {NULL, 0, DIS_HRS_HDL_IDX},

  /* Read:  DIS Firmware revision string */
  {NULL, 0, DIS_FRS_HDL_IDX},

  /* Read:  DIS Software revision string */
  {NULL, 0, DIS_SRS_HDL_IDX},

  /* Read:  DIS System ID */
  {NULL, 0, DIS_SID_HDL_IDX},

  /* Read:  DIS Registration certificate data */
  {NULL, 0, DIS_RCD_HDL_IDX},

  /* Read:  DIS PnP ID */
  {NULL, 0, DIS_PNP_ID_HDL_IDX},

  /* Read:  HRS Body sensor location */
  {NULL, 0, HRPC_HRS_BSL_HDL_IDX},

  /* Write:  HRS Control point "Reset Energy Expended"  */
  {centralSubrateHrsRstEnExp, sizeof(centralSubrateHrsRstEnExp), HRPC_HRS_HRCP_HDL_IDX + CENTRAL_SUBRATE_DISC_HRS_START },

  /* Write:  HRS Heart rate measurement CCC descriptor  */
  {centralSubrateCccNtfVal, sizeof(centralSubrateCccNtfVal), HRPC_HRS_HRM_CCC_HDL_IDX + CENTRAL_SUBRATE_DISC_HRS_START },
};

/* Characteristic configuration list length */
#define CENTRAL_SUBRATE_DISC_MASTER_CFG_LIST_LEN   (sizeof(centralSubrateDiscMasterCfgList) / sizeof(attcDiscCfg_t))

/* sanity check:  make sure configuration list length is <= handle list length */
WSF_CT_ASSERT(CENTRAL_SUBRATE_DISC_MASTER_CFG_LIST_LEN <= CENTRAL_SUBRATE_DISC_MASTER_CFG_LIST_LEN);

/**************************************************************************************************
  ATT Server Data
**************************************************************************************************/

/*! enumeration of client characteristic configuration descriptors used in local ATT server */
enum
{
  CENTRAL_SUBRATE_GATT_SC_CCC_IDX,        /*! GATT service, service changed characteristic */
  CENTRAL_SUBRATE_NUM_CCC_IDX             /*! Number of ccc's */
};

/*! client characteristic configuration descriptors settings, indexed by ccc enumeration */
static const attsCccSet_t centralSubrateCccSet[CENTRAL_SUBRATE_NUM_CCC_IDX] =
{
  /* cccd handle         value range                 security level */
  {GATT_SC_CH_CCC_HDL,   ATT_CLIENT_CFG_INDICATE,    DM_SEC_LEVEL_ENC}    /* CENTRAL_SUBRATE_GATT_SC_CCC_IDX */
};

/*************************************************************************************************/
/*!
 *  \brief  Application DM callback.
 *
 *  \param  pDmEvt  DM callback event
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralSubrateDmCback(dmEvt_t *pDmEvt)
{
  dmEvt_t *pMsg;
  uint16_t  len;
  uint16_t  reportLen;

  len = DmSizeOfEvt(pDmEvt);

  if (pDmEvt->hdr.event == DM_SCAN_REPORT_IND)
  {
    reportLen = pDmEvt->scanReport.len;
  }
  else
  {
    reportLen = 0;
  }

  if ((pMsg = WsfMsgAlloc(len + reportLen)) != NULL)
  {
    memcpy((uint8_t *)pMsg, (uint8_t *)pDmEvt, len);

    if (pDmEvt->hdr.event == DM_SCAN_REPORT_IND)
    {
      pMsg->scanReport.pData = (uint8_t *) ((uint8_t *) pMsg + len);
      memcpy(pMsg->scanReport.pData, pDmEvt->scanReport.pData, reportLen);
    }

    WsfMsgSend(centralSubrateCb.handlerId, pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Application  ATT callback.
 *
 *  \param  pEvt    ATT callback event
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralSubrateAttCback(attEvt_t *pEvt)
{
  attEvt_t *pMsg;

  if ((pMsg = WsfMsgAlloc(sizeof(attEvt_t) + pEvt->valueLen)) != NULL)
  {
    memcpy((uint8_t *)pMsg, (uint8_t *)pEvt, sizeof(attEvt_t));
    pMsg->pValue = (uint8_t *) (pMsg + 1);
    memcpy(pMsg->pValue, pEvt->pValue, pEvt->valueLen);
    WsfMsgSend(centralSubrateCb.handlerId, pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Perform actions on scan start.
 *
 *  \param  pMsg    Pointer to DM callback event message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralSubrateScanStart(dmEvt_t *pMsg)
{
  if (pMsg->hdr.status == HCI_SUCCESS)
  {
    centralSubrateCb.scan_state = CENTRAL_SUBRATE_SCANNING;
    // set it to lowest possible value of RSSI
    centralSubrateConnInfo.rssi = -128;
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Perform actions on scan stop.
 *
 *  \param  pMsg    Pointer to DM callback event message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralSubrateScanStop(dmEvt_t *pMsg)
{
  if (pMsg->hdr.status == HCI_SUCCESS)
  {
    centralSubrateCb.scan_state = CENTRAL_SUBRATE_SCAN_NONE;
    centralSubrateCb.autoConnect = FALSE;

    /* Open connection */
    if (centralSubrateConnInfo.doConnect)
    {
      AppConnOpen(centralSubrateConnInfo.addrType, centralSubrateConnInfo.addr, centralSubrateConnInfo.dbHdl);
      centralSubrateConnInfo.doConnect = FALSE;
    }
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Handle a scan report.
 *
 *  \param  pMsg    Pointer to DM callback event message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralSubrateScanReport(dmEvt_t *pMsg)
{
  uint8_t  device_name[32] = {0};
  uint32_t name_length = 0;
  uint8_t *pData;
  appDbHdl_t dbHdl;
  bool_t  connect = FALSE;

  /* disregard if not scanning or autoconnecting */
  if ( (centralSubrateCb.scan_state == CENTRAL_SUBRATE_SCAN_NONE) || !centralSubrateCb.autoConnect )
  {
    return;
  }

  pData  = DmFindAdType(DM_ADV_TYPE_LOCAL_NAME, pMsg->scanReport.len,
                        pMsg->scanReport.pData);
  if ( pData )
  {
    name_length = *pData;
    name_length--;

    memcpy(device_name, pData + 2, name_length);
  }
  else
  {
    pData = DmFindAdType(DM_ADV_TYPE_SHORT_NAME, pMsg->scanReport.len,
                          pMsg->scanReport.pData);
    if ( pData )
    {
      name_length = *pData;
      name_length--;

      memcpy(device_name, pData + 2, name_length);
    }
  }

  WsfTrace("Found: 0x%02x:%02x:%02x:%02x:%02x:%02x rssi: %d %s",
    pMsg->scanReport.addr[5],
    pMsg->scanReport.addr[4],
    pMsg->scanReport.addr[3],
    pMsg->scanReport.addr[2],
    pMsg->scanReport.addr[1],
    pMsg->scanReport.addr[0],
    pMsg->scanReport.rssi,
    device_name
    );

  /* if we already have a bond with this device then connect to it */
  if ((dbHdl = AppDbFindByAddr(pMsg->scanReport.addrType, pMsg->scanReport.addr)) != APP_DB_HDL_NONE)
  {
    /* if this is a directed advertisement where the initiator address is an RPA */
    if (DM_RAND_ADDR_RPA(pMsg->scanReport.directAddr, pMsg->scanReport.directAddrType))
    {
      /* resolve direct address to see if it's addressed to us */
      AppMasterResolveAddr(pMsg, dbHdl, APP_RESOLVE_DIRECT_RPA);
    }
    else
    {
      connect = TRUE;
    }
  }
  /* if the peer device uses an RPA */
  else if (DM_RAND_ADDR_RPA(pMsg->scanReport.addr, pMsg->scanReport.addrType))
  {
    /* resolve advertiser's RPA to see if we already have a bond with this device */
    AppMasterResolveAddr(pMsg, APP_DB_HDL_NONE, APP_RESOLVE_ADV_RPA);
  }
  else if ((name_length > 0) && !strncmp((char *)device_name, "Periph_Sub", name_length))
  {
    connect = TRUE;
  }

  if (connect)
  {
    if (pMsg->scanReport.rssi > centralSubrateConnInfo.rssi)
    {
      centralSubrateConnInfo.rssi = pMsg->scanReport.rssi;

      /* connect will start after scanning stops with expiration
       * of scan timer.
       */

      /* Store peer information for connect on scan stop */
      centralSubrateConnInfo.addrType = DmHostAddrType(pMsg->scanReport.addrType);
      memcpy(centralSubrateConnInfo.addr, pMsg->scanReport.addr, sizeof(bdAddr_t));
      centralSubrateConnInfo.dbHdl = dbHdl;
      centralSubrateConnInfo.doConnect = TRUE;
    }
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Application ATTS client characteristic configuration callback.
 *
 *  \param  pDmEvt  DM callback event
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralSubrateCccCback(attsCccEvt_t *pEvt)
{
  attsCccEvt_t  *pMsg;
  appDbHdl_t    dbHdl;

  /* If CCC not set from initialization and there's a device record and currently bonded */
  if ((pEvt->handle != ATT_HANDLE_NONE) &&
      ((dbHdl = AppDbGetHdl((dmConnId_t) pEvt->hdr.param)) != APP_DB_HDL_NONE) &&
      AppCheckBonded((dmConnId_t)pEvt->hdr.param))
  {
    /* Store value in device database. */
    AppDbSetCccTblValue(dbHdl, pEvt->idx, pEvt->value);
  }

  if ((pMsg = WsfMsgAlloc(sizeof(attsCccEvt_t))) != NULL)
  {
    memcpy((uint8_t *)pMsg, (uint8_t *)pEvt, sizeof(attsCccEvt_t));
    WsfMsgSend(centralSubrateCb.handlerId, pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Perform UI actions on connection open.
 *
 *  \param  pMsg    Pointer to DM callback event message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralSubrateOpen(dmEvt_t *pMsg)
{

}

/*************************************************************************************************/
/*!
 *  \brief  Set up advertising and other procedures that need to be performed after
 *          device reset.
 *
 *  \param  pMsg    Pointer to DM callback event message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralSubrateSetup(dmEvt_t *pMsg)
{
  centralSubrateCb.autoConnect = TRUE;
  centralSubrateConnInfo.doConnect = FALSE;
  centralSubrateCb.scan_state = CENTRAL_SUBRATE_SCAN_START;
  AppScanStart(centralSubrateMasterCfg.discMode, centralSubrateMasterCfg.scanType,
               centralSubrateMasterCfg.scanDuration);

  DmConnSetConnSpec((hciConnSpec_t *) &centralSubrateConnCfg);
}

/*************************************************************************************************/
/*!
 *  \brief  Process a received ATT read response, notification, or indication when connected in
 *          the master role to a heart rate profile sensor.
 *
 *  \param  pMsg    Pointer to ATT callback event message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralSubrateMasterValueUpdate(attEvt_t *pMsg)
{
  if (pMsg->hdr.status == ATT_SUCCESS)
  {
    /* determine which profile the handle belongs to; start with most likely */

    /* heart rate */
    if (HrpcHrsValueUpdate(pCentralSubrateHrsHdlList, pMsg) == ATT_SUCCESS)
    {
      return;
    }

    /* device information */
    if (DisValueUpdate(pCentralSubrateDisHdlList, pMsg) == ATT_SUCCESS)
    {
      return;
    }

    /* GATT */
    if (GattValueUpdate(pCentralSubrateMstGattHdlList, pMsg) == ATT_SUCCESS)
    {
      return;
    }
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Loop through the open connection ID list and find the first connection for the given role
 *
 *  \param  pConnIdList    Buffer with the connection IDs (must be DM_CONN_MAX bytes).
 *  \param  role           Role to look for.
 *
 *  \return None.
 */
/*************************************************************************************************/
static dmConnId_t centralSubrateGetConnIdByRole(dmConnId_t *pConnIdList, uint8_t role)
{
  uint8_t i;

  for (i = 0; i < DM_CONN_MAX; i++)
  {
    if (pConnIdList[i] != DM_CONN_ID_NONE)
    {
      if (DmConnRole(pConnIdList[i]) == role)
      {
        return pConnIdList[i];
      }
    }
  }

  return DM_CONN_ID_NONE;
}

/*************************************************************************************************/
/*!
 *  \brief  Button press callback.
 *
 *  \param  btn    Button press.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralSubrateBtnCback(uint8_t btn)
{
  dmConnId_t      masterConnId;
  dmConnId_t      connIdList[DM_CONN_MAX];

  AppConnOpenList(connIdList);

  masterConnId = centralSubrateGetConnIdByRole(connIdList, DM_ROLE_MASTER);

  APP_TRACE_INFO2("btn: %d - master conn id: %d", btn, masterConnId);

  /* No connection as a master */
  switch (btn)
  {
    case APP_UI_BTN_1_SHORT:
      /* if scanning cancel scanning */
      if ( centralSubrateCb.scan_state != CENTRAL_SUBRATE_SCAN_NONE )
      {
        AppScanStop();
      }
      /* else auto connect */
      else if (!centralSubrateCb.autoConnect)
      {
        centralSubrateCb.autoConnect = TRUE;
        centralSubrateConnInfo.doConnect = FALSE;
        centralSubrateCb.scan_state = CENTRAL_SUBRATE_SCAN_START;

        AppScanStart(centralSubrateMasterCfg.discMode, centralSubrateMasterCfg.scanType,
                      centralSubrateMasterCfg.scanDuration);
      }
      return;

    case APP_UI_BTN_1_LONG:
      /* clear bonded device info */
      AppClearAllBondingInfo();
      return;

    default:
      break;
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Discovery callback.
 *
 *  \param  connId    Connection identifier.
 *  \param  status    Service or configuration status.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralSubrateDiscCback(dmConnId_t connId, uint8_t status)
{
  switch(status)
  {
    case APP_DISC_INIT:
      AppDiscSetHdlList(connId, CENTRAL_SUBRATE_DISC_MASTER_HDL_LIST_LEN, centralSubrateCb.hdlMasterList);
      break;

    case APP_DISC_READ_DATABASE_HASH:
      /* Read peer's database hash */
      AppDiscReadDatabaseHash(connId);
      break;

    case APP_DISC_SEC_REQUIRED:
      /* request security */
      AppMasterSecurityReq(connId);
      break;

    case APP_DISC_START:
      /* initialize discovery state */
      centralSubrateCb.discState = CENTRAL_SUBRATE_DISC_MASTER_GATT_SVC;
      GattDiscover(connId, pCentralSubrateMstGattHdlList);
      break;

    case APP_DISC_FAILED:
    case APP_DISC_CMPL:
      /* next discovery state */
      centralSubrateCb.discState++;
      if (centralSubrateCb.discState == CENTRAL_SUBRATE_DISC_MASTER_DIS_SVC)
      {
        /* discover device information service */
        DisDiscover(connId, pCentralSubrateDisHdlList);
      }
      else if (centralSubrateCb.discState == CENTRAL_SUBRATE_DISC_MASTER_HRS_SVC)
      {
        /* discover heart rate service */
        HrpcHrsDiscover(connId, pCentralSubrateHrsHdlList);
      }
      else
      {
        /* discovery complete */
        AppDiscComplete(connId, APP_DISC_CMPL);

        /* start configuration */
        AppDiscConfigure(connId, APP_DISC_CFG_START, CENTRAL_SUBRATE_DISC_MASTER_CFG_LIST_LEN,
                          (attcDiscCfg_t *) centralSubrateDiscMasterCfgList,
                          CENTRAL_SUBRATE_DISC_MASTER_HDL_LIST_LEN, centralSubrateCb.hdlMasterList);
      }
      break;

    case APP_DISC_CFG_START:
      /* start configuration */
      AppDiscConfigure(connId, APP_DISC_CFG_START, CENTRAL_SUBRATE_DISC_MASTER_CFG_LIST_LEN,
                      (attcDiscCfg_t *) centralSubrateDiscMasterCfgList,
                      CENTRAL_SUBRATE_DISC_MASTER_HDL_LIST_LEN, centralSubrateCb.hdlMasterList);
      break;

    case APP_DISC_CFG_CMPL:
      AppDiscComplete(connId, status);
      break;

    case APP_DISC_CFG_CONN_START:
      break;

    default:
      break;
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Process messages from the event handler.
 *
 *  \param  pMsg    Pointer to message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void centralSubrateProcMsg(dmEvt_t *pMsg)
{
  uint8_t uiEvent = APP_UI_NONE;

  switch(pMsg->hdr.event)
  {
    case ATTC_READ_RSP:
    case ATTC_HANDLE_VALUE_NTF:
    case ATTC_HANDLE_VALUE_IND:
      centralSubrateMasterValueUpdate((attEvt_t *) pMsg);
      break;

    case ATT_MTU_UPDATE_IND:
      APP_TRACE_INFO1("Negotiated MTU %d", ((attEvt_t *)pMsg)->mtu);
      break;

    case DM_RESET_CMPL_IND:
      // set database hash calculating status to true until a new hash is generated after reset
      attsCsfSetHashUpdateStatus(TRUE);

      // Generate ECC key if configured support secure connection,
      // else will calcualte ATT database hash
      if ( centralSubrateSecCfg.auth & DM_AUTH_SC_FLAG )
      {
          DmSecGenerateEccKeyReq();
      }
      else
      {
          AttsCalculateDbHash();
      }
      uiEvent = APP_UI_RESET_CMPL;
      break;

    case ATTS_DB_HASH_CALC_CMPL_IND:
      centralSubrateSetup(pMsg);
      break;

    case DM_SEC_ECC_KEY_IND:
      DmSecSetEccKey(&pMsg->eccMsg.data.key);
      // Only calculate database hash if the calculating status is in progress
      if ( attsCsfGetHashUpdateStatus() )
      {
        AttsCalculateDbHash();
      }
      break;

    case DM_SCAN_START_IND:
    {
      centralSubrateScanStart(pMsg);
      uiEvent = APP_UI_SCAN_START;

      hciLeSetDefaultSubrate_t subrate;
      subrate.subrateMin = centralSubrateDefaultSubrateCfg.subrateMin;
      subrate.subrateMax = centralSubrateDefaultSubrateCfg.subrateMax;
      subrate.maxLatency = centralSubrateDefaultSubrateCfg.maxLatency;
      subrate.contNum = centralSubrateDefaultSubrateCfg.contNum;
      subrate.supTimeout = centralSubrateDefaultSubrateCfg.supTimeout;

      APP_TRACE_INFO0("Setting default connection subrate parameters");
      APP_TRACE_INFO1("- Min Subrate Factor: %d", subrate.subrateMin);
      APP_TRACE_INFO1("- Max Subrate Factor: %d", subrate.subrateMax);
      APP_TRACE_INFO1("- Max Peripheral Latency: %d", subrate.maxLatency);
      APP_TRACE_INFO1("- Continuation Number: %d", subrate.contNum);
      APP_TRACE_INFO1("- Supervision Timeout: %d", subrate.supTimeout);
      HciLeSetDefaultSubrateCmd(&subrate);
    }
      break;

    case DM_SCAN_STOP_IND:
      centralSubrateScanStop(pMsg);
      uiEvent = APP_UI_SCAN_STOP;
      break;

    case DM_SCAN_REPORT_IND:
      centralSubrateScanReport(pMsg);
      break;

    case DM_CONN_OPEN_IND:
      centralSubrateOpen(pMsg);
      uiEvent = APP_UI_CONN_OPEN;
      break;

    case DM_CONN_CLOSE_IND:
      uiEvent = APP_UI_CONN_CLOSE;
      break;

    case DM_PHY_UPDATE_IND:
      APP_TRACE_INFO3("DM_PHY_UPDATE_IND status: %d, RX: %d, TX: %d", pMsg->phyUpdate.status, pMsg->phyUpdate.rxPhy, pMsg->phyUpdate.txPhy);
      break;

    case DM_SEC_PAIR_CMPL_IND:
      DmSecGenerateEccKeyReq();
      uiEvent = APP_UI_SEC_PAIR_CMPL;
      break;

    case DM_SEC_PAIR_FAIL_IND:
      DmSecGenerateEccKeyReq();
      uiEvent = APP_UI_SEC_PAIR_FAIL;
      break;

    case DM_SEC_ENCRYPT_IND:
      uiEvent = APP_UI_SEC_ENCRYPT;
      break;

    case DM_SEC_ENCRYPT_FAIL_IND:
      uiEvent = APP_UI_SEC_ENCRYPT_FAIL;
      break;

    case DM_SEC_AUTH_REQ_IND:
      AppHandlePasskey(&pMsg->authReq);
      break;

    case DM_PRIV_CLEAR_RES_LIST_IND:
      APP_TRACE_INFO1("Clear resolving list status 0x%02x", pMsg->hdr.status);
      break;

    case DM_VENDOR_SPEC_CMD_CMPL_IND:
      {
        #if defined(AM_PART_APOLLO) || defined(AM_PART_APOLLO2)

          uint8_t *param_ptr = &pMsg->vendorSpecCmdCmpl.param[0];

          switch (pMsg->vendorSpecCmdCmpl.opcode)
          {
            case 0xFC20: //read at address
            {
              uint32_t read_value;

              BSTREAM_TO_UINT32(read_value, param_ptr);

              APP_TRACE_INFO3("VSC 0x%0x complete status %x param %x",
                pMsg->vendorSpecCmdCmpl.opcode,
                pMsg->hdr.status,
                read_value);
            }

            break;
            default:
                APP_TRACE_INFO2("VSC 0x%0x complete status %x",
                    pMsg->vendorSpecCmdCmpl.opcode,
                    pMsg->hdr.status);
            break;
          }

        #endif
      }
      break;

    default:
      break;
  }

  if (uiEvent != APP_UI_NONE)
  {
    AppUiAction(uiEvent);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Application handler init function called during system initialization.
 *
 *  \param  handlerID  WSF handler ID.
 *
 *  \return None.
 */
/*************************************************************************************************/
void CentralSubrateHandlerInit(wsfHandlerId_t handlerId)
{
  APP_TRACE_INFO0("CentralSubrateHandlerInit");

  /* store handler ID */
  centralSubrateCb.handlerId = handlerId;

  /* Set configuration pointers */
  pAppMasterCfg = (appMasterCfg_t *) &centralSubrateMasterCfg;
  pAppSecCfg = (appSecCfg_t *) &centralSubrateSecCfg;
  pAppUpdateCfg = (appUpdateCfg_t *) &centralSubrateUpdateCfg;
  pAppDiscCfg = (appDiscCfg_t *) &centralSubrateDiscCfg;
  pAppCfg = (appCfg_t *) &centralSubrateAppCfg;

  /* Set stack configuration pointers */
  pSmpCfg = (smpCfg_t *) &centralSubrateSmpCfg;
#ifdef AM_BLE_EATT
  pEattCfg = (eattCfg_t *) &centralSubrateeattCfg;
#endif

  /* Initialize application framework */
  AppMasterInit();
  AppDiscInit();
}

/*************************************************************************************************/
/*!
 *  \brief  WSF event handler for application.
 *
 *  \param  event   WSF event mask.
 *  \param  pMsg    WSF message.
 *
 *  \return None.
 */
/*************************************************************************************************/
void CentralSubrateHandler(wsfEventMask_t event, wsfMsgHdr_t *pMsg)
{
  if (pMsg != NULL)
  {
    APP_TRACE_INFO1("CentralSubrate got evt %d", pMsg->event);

    /* process ATT messages */
    if (pMsg->event <= ATT_CBACK_END)
    {
      /* process discovery-related ATT messages */
      AppDiscProcAttMsg((attEvt_t *) pMsg);

      /* process server-related ATT messages */
      AppServerProcAttMsg(pMsg);
    }
    /* process DM messages */
    else if (pMsg->event <= DM_CBACK_END)
    {
      /* process advertising and connection-related messages */
      AppMasterProcDmMsg((dmEvt_t *) pMsg);

      /* process security-related messages */
      AppMasterSecProcDmMsg((dmEvt_t *) pMsg);

      /* process discovery-related messages */
      AppDiscProcDmMsg((dmEvt_t *) pMsg);
    }

    /* perform profile and user interface-related operations */
    centralSubrateProcMsg((dmEvt_t *) pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Start the application.
 *
 *  \return None.
 */
/*************************************************************************************************/
void CentralSubrateStart(void)
{
  /* EATT initialization */
#ifdef AM_BLE_EATT
  EattInit(EATT_ROLE_INITIATOR | EATT_ROLE_ACCEPTOR);
#endif

  /* Register for stack callbacks */
  DmRegister(centralSubrateDmCback);
  DmConnRegister(DM_CLIENT_ID_APP, centralSubrateDmCback);
  AttRegister(centralSubrateAttCback);
  AttConnRegister(AppServerConnCback);
  AttsCccRegister(CENTRAL_SUBRATE_NUM_CCC_IDX, (attsCccSet_t *) centralSubrateCccSet, centralSubrateCccCback);

  /* Register for app framework button callbacks */
  AppUiBtnRegister(centralSubrateBtnCback);

  /* Register for app framework discovery callbacks */
  AppDiscRegister(centralSubrateDiscCback);

  /* Initialize attribute server database */
  SvcCoreAddGroup();

  /* Reset the device */
  DmDevReset();
}
