//*****************************************************************************
//
//! @file pawr_adv_main.c
//!
//! @brief Ambiq Micro's demonstration of Periodic Advertising with Response(Advertiser)
//
//*****************************************************************************

//*****************************************************************************
//
// Copyright (c) 2026, Ambiq Micro, Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the copyright holder nor the names of its
// contributors may be used to endorse or promote products derived from this
// software without specific prior written permission.
//
// Third party software included in this distribution is subject to the
// additional license terms as defined in the /docs/licenses directory.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// This is part of revision stable-e24d618f43 of the AmbiqSuite Development Package.
//
//*****************************************************************************

#include <string.h>
#include "wsf_types.h"
#include "util/bstream.h"
#include "wsf_msg.h"
#include "wsf_trace.h"
#include "hci_api.h"
#include "dm_api.h"
#include "gap/gap_api.h"
#include "att_api.h"
#include "smp_api.h"
#include "app_api.h"
#include "app_db.h"
#include "app_ui.h"
#include "app_hw.h"
#include "app_main.h"
#include "svc_ch.h"
#include "svc_core.h"
#include "gatt/gatt_api.h"
#include "pawr_adv_api.h"
#include "atts_main.h"
#include "am_util_stdio.h"
#include "am_mcu_apollo.h"

/**************************************************************************************************
  Macros
**************************************************************************************************/
#define NUM_SUB_EVENT                   (5)
#define SUB_EVT_PACKET_SIZE             (5)
#define NUM_RSP_SLOTS                   (5)
#define ADV_BUF_LEN_MAX                 (252 * 5)

/**************************************************************************************************
  Data Types
**************************************************************************************************/
/*! application control block */
struct
{
  uint16_t          hdlList[DM_CONN_MAX][APP_DB_HDL_LIST_LEN];   /*! Cached handle list */
  wsfHandlerId_t    handlerId;                      /*! WSF hander ID */
  bool_t            autoConnect;                    /*! TRUE if auto-connecting */
  uint8_t           discState[DM_CONN_MAX];         /*! Service discovery state */
  uint8_t           hdlListLen;                     /*! Cached handle list length */
  uint8_t           btnConnId;                      /*! The index of the connection ID for button presses */
} pawrAdvCb;

/*! connection control block */
typedef struct
{
  appDbHdl_t          dbHdl;                        /*! Device database record handle type */
  uint8_t             addrType;                     /*! Type of address of device to connect to */
  bdAddr_t            addr;                         /*! Address of device to connect to */
  bool_t              doConnect;                    /*! TRUE to issue connect on scan complete */
  uint8_t             secPhy;                       /*! Secondary Advertising channel from extended advertising rerport event*/
  uint16_t            eventType;                    /*! event type from extended advertising report event*/
  uint8_t             conn_id;
  dmSyncId_t          syncId;
} pawrAdvConnInfo_t;

/*! Subvet parameter */
typedef struct
{
  uint8_t numSubEvt;
  uint8_t subEvtInt;
  uint8_t rspSlotDelay;
  uint8_t rspSlotSpacing;
  uint8_t numRspSlots;
} pawrAdvSubEvtParam_t;

/*! Application message type */
typedef union
{
  wsfMsgHdr_t     hdr;
  dmEvt_t         dm;
  attsCccEvt_t    ccc;
  attEvt_t        att;
} pawrAdvMsg_t;


typedef struct
{
  hciExtAdvParam_t advParam;
  uint8_t advType;
  bool_t useLegAdvPdu;
} extAdvParam_t;

/**************************************************************************************************
  Configurable Parameters
**************************************************************************************************/
appExtAdvCfg_t pawrAdvAdvCfg =
{
#if (DM_NUM_ADV_SETS == 1)
  {0},
  {800},
  {0},
  {FALSE},
  {0},
  #elif (DM_NUM_ADV_SETS > 1)
  {0, 0},
  {800, 800},
  {0, 0},
  {FALSE, FALSE},
  {800, 800},
#endif
};

/**************************************************************************************************
  Global Variables
**************************************************************************************************/
static uint8_t subEvtDataStore[NUM_SUB_EVENT][SUB_EVT_PACKET_SIZE];
static uint8_t subEvtDataBufs[NUM_SUB_EVENT*SUB_EVT_PACKET_SIZE];
static uint8_t adv_handle[DM_NUM_ADV_SETS] = {0};

static pawrAdvSubEvtParam_t pawrAdvSubEvtParam =
{
  .numSubEvt = NUM_SUB_EVENT,
  .subEvtInt = 0x30,
  .rspSlotDelay = 5,
  .rspSlotSpacing = 0x50,
  .numRspSlots = 5,
};

/*! advertising data, discoverable mode */
static const uint8_t pawrAdvAdvDataDisc_per[] =
{
  /*! flags */
  2,                                      /*! length */
  DM_ADV_TYPE_FLAGS,                      /*! AD type */
  DM_FLAG_LE_GENERAL_DISC |               /*! flags */
  DM_FLAG_LE_BREDR_NOT_SUP,

  /*! device name */
  8,                                      /*! length */
  DM_ADV_TYPE_LOCAL_NAME,                 /*! AD type */
  'A',
  'm',
  '_',
  'P',
  'A',
  'w',
  'R'
};

/*! configurable parameters for slave */
static const appSlaveCfg_t pawrAdvSlaveCfg =
{
  PER_ADV_RSP_CONN_MAX,                           /*! Maximum connections */
};

/*! configurable parameters for master extended scan */
static const appExtMasterCfg_t pawrAdvMasterExtCfg =
{
  {96},                                      /*! The scan interval, in 0.625 ms units */
  {48},                                      /*! The scan window, in 0.625 ms units  */
  0,                                         /*! The scan duration in ms */
  0,                                         /*! Scan period*/
  DM_DISC_MODE_GENERAL,                       /*! The GAP discovery mode */
  {DM_SCAN_TYPE_ACTIVE}                      /*! The scan type (active or passive) */
};

/*! configurable parameters for security */
static const appSecCfg_t pawrAdvSecCfg =
{
  DM_AUTH_SC_FLAG,      /*! Authentication and bonding flags */
  0,                    /*! Initiator key distribution flags */
  DM_KEY_DIST_LTK,      /*! Responder key distribution flags */
  FALSE,                /*! TRUE if Out-of-band pairing data is present */
  FALSE                 /*! TRUE to initiate security upon connection */
};

/*! configurable parameters for connection parameter update */
static const appUpdateCfg_t pawrAdvUpdateCfg =
{
  3000,                 /*! Connection idle period in ms before attempting
                            connection parameter update; set to zero to disable */
  48,                   /*! Minimum connection interval in 1.25ms units */
  60,                   /*! Maximum connection interval in 1.25ms units */
  4,                    /*! Connection latency */
  600,                  /*! Supervision timeout in 10ms units */
  5                     /*! Number of update attempts before giving up */
};

/*! SMP security parameter configuration */
static const smpCfg_t pawrAdvSmpCfg =
{
  3000,                                   /*! 'Repeated attempts' timeout in msec */
  SMP_IO_NO_IN_NO_OUT,                    /*! I/O Capability */
  7,                                      /*! Minimum encryption key length */
  16,                                     /*! Maximum encryption key length */
  3,                                      /*! Attempts to trigger 'repeated attempts' timeout */
  0,                                      /*! Device authentication requirements */
  64000,                                  /*! Maximum repeated attempts timeout in msec */
  64000,                                  /*! Time msec before attemptExp decreases */
  2                                       /*! Repeated attempts multiplier exponent */
};

/*! ATT configurable parameters (increase MTU) */
static const attCfg_t pawrAdvAttCfg =
{
  15,                               /* ATT server service discovery connection idle timeout in seconds */
  23,                               /* desired ATT MTU */
  ATT_MAX_TRANS_TIMEOUT,            /* transcation timeout in seconds */
  4                                 /* number of queued prepare writes supported by server */
};

static uint8_t peerAddr[BDA_ADDR_LEN] = {0x12, 0x34, 0x56, 0x78, 0xAB, 0xFF};

extAdvParam_t  extAdvParam[DM_NUM_ADV_SETS] =
{
  {
    {
      .priAdvInterMin  = 1000 / 0.625,      // 1s minimum interval
      .priAdvInterMax  = 1000 / 0.625,
      .priAdvChanMap   = DM_ADV_CHAN_ALL,
      .ownAddrType     = DM_ADDR_PUBLIC,
      .peerAddrType    = DM_ADDR_PUBLIC,
      .pPeerAddr       = peerAddr,
      .advFiltPolicy   = HCI_ADV_FILT_NONE,
      .advTxPwr        = HCI_TX_PWR_NO_PREFERENCE,
      .priAdvPhy       = HCI_ADV_PHY_LE_1M,
      .secAdvMaxSkip   = 0,
      .secAdvPhy       = HCI_ADV_PHY_LE_1M,
      .advSID          = 0,
      .scanReqNotifEna = FALSE,
    },
    .advType           = DM_ADV_CONN_UNDIRECT,
    .useLegAdvPdu      = FALSE,
  },

  {
    {
      .priAdvInterMin  = 1000 / 0.625,    // 1s minimum interval
      .priAdvInterMax  = 1000 / 0.625,
      .priAdvChanMap   = DM_ADV_CHAN_ALL,
      .ownAddrType     = DM_ADDR_PUBLIC,
      .peerAddrType    = DM_ADDR_PUBLIC,
      .pPeerAddr       = peerAddr,
      .advFiltPolicy   = HCI_ADV_FILT_NONE,
      .advTxPwr        = HCI_TX_PWR_NO_PREFERENCE,
      .priAdvPhy       = HCI_ADV_PHY_LE_1M,
      .secAdvMaxSkip   = 0,
      .secAdvPhy       = HCI_ADV_PHY_LE_2M,
      .advSID          = 1,
      .scanReqNotifEna = FALSE,
    },
    .advType           = DM_EXT_ADV_CONN_UNDIRECT,
    .useLegAdvPdu      = FALSE,
  }
};

/*! Connection parameters */
static const hciConnSpec_t connection_para =
{
  0x0220,                                     /*! Minimum connection interval in 1.25ms units */
  0x0220,                                     /*! Maximum connection interval in 1.25ms units */
  0,                                          /*! Connection latency */
  0x0C80,                                     /*! Supervision timeout in 10ms units */
  0,                                          /*! Unused */
  0                                           /*! Unused */
};
/*! connection info */
static pawrAdvConnInfo_t pawrAdvConnInfo;

/*************************************************************************************************/
/*!
 *  \brief  Application DM callback.
 *
 *  \param  pDmEvt  DM callback event
 *
 *  \return None.
 */
/*************************************************************************************************/
static void pawrAdvDmCback(dmEvt_t *pDmEvt)
{
  dmEvt_t   *pMsg;
  uint16_t  len;

  len = DmSizeOfEvt(pDmEvt);

  if ((pMsg = WsfMsgAlloc(len)) != NULL)
  {
    memcpy((uint8_t *)pMsg, (uint8_t *)pDmEvt, len);
    WsfMsgSend(pawrAdvCb.handlerId, pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Application ATT callback.
 *
 *  \param  pEvt    ATT callback event
 *
 *  \return None.
 */
/*************************************************************************************************/
static void pawrAdvAttCback(attEvt_t *pEvt)
{
  attEvt_t *pMsg;

  if ((pMsg = WsfMsgAlloc(sizeof(attEvt_t) + pEvt->valueLen)) != NULL)
  {
    memcpy(pMsg, pEvt, sizeof(attEvt_t));
    pMsg->pValue = (uint8_t *) (pMsg + 1);
    memcpy(pMsg->pValue, pEvt->pValue, pEvt->valueLen);
    WsfMsgSend(pawrAdvCb.handlerId, pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Set up advertising parameters used in Extended Advertising
 *
 *  \param  NULL.
 *
 *  \return None.
 */
/*************************************************************************************************/
void AppExtSetParam(void)
{
  uint8_t advHandle = 0;

  for ( advHandle = 0; advHandle < DM_NUM_ADV_SETS; advHandle++ )
  {
    // set extended advertising interval
    pawrAdvAdvCfg.advInterval[advHandle] = (uint16_t)(extAdvParam[advHandle].advParam.priAdvInterMin);

    DmAdvSetChannelMap(advHandle, extAdvParam[advHandle].advParam.priAdvChanMap);

    DmAdvSetAddrType(extAdvParam[advHandle].advParam.ownAddrType);

    AppExtSetAdvPeerAddr(advHandle, extAdvParam[advHandle].advParam.peerAddrType,
                        extAdvParam[advHandle].advParam.pPeerAddr);

    DmDevSetExtFilterPolicy(advHandle, DM_FILT_POLICY_MODE_ADV, HCI_ADV_FILT_NONE);

    DmAdvIncTxPwr(advHandle, FALSE, extAdvParam[advHandle].advParam.advTxPwr);

    DmAdvSetPhyParam(advHandle, extAdvParam[advHandle].advParam.priAdvPhy,
                        extAdvParam[advHandle].advParam.secAdvMaxSkip, extAdvParam[advHandle].advParam.secAdvPhy);

    DmAdvScanReqNotifEnable(advHandle, extAdvParam[advHandle].advParam.scanReqNotifEna);

    AppExtSetAdvType(advHandle, extAdvParam[advHandle].advType);

    DmAdvUseLegacyPdu(advHandle, extAdvParam[advHandle].useLegAdvPdu);
    pawrAdvAdvCfg.useLegacyPdu[advHandle] = extAdvParam[advHandle].useLegAdvPdu;
  }
}

void pawrAdvSubEvtDataInit(void)
{
  uint8_t *p_buf = subEvtDataBufs;
  uint8_t len    = 0;

  uint8_t arry_size = sizeof(subEvtDataStore) / sizeof((subEvtDataStore)[0]);
  for (uint8_t i = 0; i < arry_size; i++)
  {
    subEvtDataStore[i][0] = sizeof(subEvtDataStore[i]) / sizeof((subEvtDataStore[i])[0]);
    subEvtDataStore[i][1] = DM_ADV_TYPE_MANUFACTURER;
    subEvtDataStore[i][2] = 0xAC;       // Ambiq company ID 0x09AC
    subEvtDataStore[i][3] = 0x09;
    subEvtDataStore[i][4] = i + 0x30;   // Use tha last byte to distinguish the subevent data

    memcpy(p_buf + len, subEvtDataStore[i], SUB_EVT_PACKET_SIZE);

    len += subEvtDataStore[i][0];
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Set up periodic advertising subevent
 *          device reset.
 *
 *  \param  pMsg    Pointer to message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void pawrAdvSubEvtSetup(pawrAdvMsg_t *pMsg)
{
  APP_TRACE_INFO0("PAwR Adverister subevt setup");

  pawrAdvSubEvtDataInit();

  pawrAdvCb.autoConnect = FALSE;
  pawrAdvConnInfo.doConnect = FALSE;

  AppExtSetParam();
  AppExtSetAdvType(adv_handle[0], DM_ADV_NONCONN_UNDIRECT);
  AppExtAdvSetData(adv_handle[0], APP_ADV_DATA_DISCOVERABLE, sizeof(pawrAdvAdvDataDisc_per),
                   (uint8_t *) pawrAdvAdvDataDisc_per, ADV_BUF_LEN_MAX);

  DmAdvSetPerSbuEvtParam(adv_handle[0], NUM_SUB_EVENT, pawrAdvSubEvtParam.subEvtInt,
                         pawrAdvSubEvtParam.rspSlotDelay, pawrAdvSubEvtParam.rspSlotSpacing,
                         pawrAdvSubEvtParam.numRspSlots);

  AppExtAdvStart(1, &adv_handle[0], APP_MODE_AUTO_INIT);
  AppPerAdvStart(adv_handle[0]);
  DmConnSetConnSpec((hciConnSpec_t *) &connection_para);
}

/*************************************************************************************************/
/*!
 *  \brief  Process messages from the event handler.
 *
 *  \param  pMsg    Pointer to message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void pawrAdvProcMsg(pawrAdvMsg_t *pMsg)
{
  uint8_t uiEvent = APP_UI_NONE;

  switch(pMsg->hdr.event)
  {
    case ATT_MTU_UPDATE_IND:
      APP_TRACE_INFO1("Negotiated MTU %d", ((attEvt_t *)pMsg)->mtu);
      break;

    case DM_RESET_CMPL_IND:
      // set database hash calculating status to true until a new hash is generated after reset
      attsCsfSetHashUpdateStatus(TRUE);

      // Generate ECC key if configured support secure connection,
      // else will calcualte ATT database hash
      if ( pawrAdvSecCfg.auth & DM_AUTH_SC_FLAG )
      {
          DmSecGenerateEccKeyReq();
      }
      else
      {
          AttsCalculateDbHash();
      }

      uiEvent = APP_UI_RESET_CMPL;
      break;

    case ATTS_DB_HASH_CALC_CMPL_IND:
      pawrAdvSubEvtSetup(pMsg);
      break;

    case DM_ADV_SET_START_IND:
      uiEvent = APP_UI_ADV_SET_START_IND;

      break;

    case DM_ADV_SET_STOP_IND:

      uiEvent = APP_UI_ADV_SET_STOP_IND;
    break;

    case DM_ADV_START_IND:
        uiEvent = APP_UI_ADV_START;
    break;

    case DM_ADV_STOP_IND:
      uiEvent = APP_UI_ADV_STOP;
      break;

    case DM_PER_ADV_RSP_REPORT_IND:
      APP_TRACE_INFO2("[PER_ADV_RSP][RX] RSSI: %d, data length: %d",
                      pMsg->dm.perAdvRspReport.perAdvRsp.rssi, pMsg->dm.perAdvRspReport.perAdvRsp.dataLen);
      if (pMsg->dm.perAdvRspReport.perAdvRsp.dataLen > 0)
      {
        am_util_stdio_printf("[PER_ADV_RSP][RX] data: ");
        for (uint8_t i = 0; i < pMsg->dm.perAdvRspReport.perAdvRsp.dataLen; i++)
        {
          am_util_stdio_printf("%02X ", *(pMsg->dm.perAdvRspReport.perAdvRsp.pData + i));
        }
        am_util_stdio_printf("\n");
        uint8_t adv_Handle = pMsg->dm.perAdvRspReport.advHandle;
        uint8_t subEvt = pMsg->dm.perAdvRspReport.subEvent;
        if (*(pMsg->dm.perAdvRspReport.perAdvRsp.pData + 1) == DM_ADV_TYPE_BD_ADDR)
        {
          memcpy(pawrAdvConnInfo.addr, (pMsg->dm.perAdvRspReport.perAdvRsp.pData + 2), sizeof(pawrAdvConnInfo.addr));
          pawrAdvConnInfo.addrType = *(pMsg->dm.perAdvRspReport.perAdvRsp.pData + 8);
          if ( AppConnIsOpen() == DM_CONN_ID_NONE )
          {
            APP_TRACE_INFO0("Setting connection parameters and initiating connection");
            DmConnSetConnParam(adv_Handle, subEvt);
            // Use only LE 1M PHY for initial connection from periodic advertising
            appConnOpen(HCI_INIT_PHY_LE_1M_BIT, pawrAdvConnInfo.addrType, pawrAdvConnInfo.addr, pawrAdvConnInfo.dbHdl);
          }
        }
      }
      break;

    case DM_CONN_OPEN_IND:
      uiEvent = APP_UI_CONN_OPEN;
      break;

    case DM_CONN_CLOSE_IND:
    {
      APP_TRACE_INFO1("conn close reason = 0x%x\n", pMsg->dm.connClose.reason);
      uiEvent = APP_UI_CONN_CLOSE;
    }
    break;

    case DM_PHY_UPDATE_IND:
      APP_TRACE_INFO3("DM_PHY_UPDATE_IND status: %d, RX: %d, TX: %d", pMsg->dm.phyUpdate.status,
                      pMsg->dm.phyUpdate.rxPhy, pMsg->dm.phyUpdate.txPhy);
      break;

    case DM_SEC_PAIR_CMPL_IND:
      DmSecGenerateEccKeyReq();
      uiEvent = APP_UI_SEC_PAIR_CMPL;
      break;

    case DM_SEC_PAIR_FAIL_IND:
      DmSecGenerateEccKeyReq();
      uiEvent = APP_UI_SEC_PAIR_FAIL;
      break;

    case DM_SEC_ENCRYPT_IND:
      uiEvent = APP_UI_SEC_ENCRYPT;
      break;

    case DM_SEC_ENCRYPT_FAIL_IND:
      uiEvent = APP_UI_SEC_ENCRYPT_FAIL;
      break;

    case DM_SEC_AUTH_REQ_IND:
      AppHandlePasskey(&pMsg->dm.authReq);
      break;

    case DM_SEC_ECC_KEY_IND:
      DmSecSetEccKey(&pMsg->dm.eccMsg.data.key);
      // Only calculate database hash if the calculating status is in progress
      if ( attsCsfGetHashUpdateStatus() )
      {
        AttsCalculateDbHash();
      }
      break;

    case DM_SEC_COMPARE_IND:
      AppHandleNumericComparison(&pMsg->dm.cnfInd);
      break;

    case DM_PRIV_CLEAR_RES_LIST_IND:
      APP_TRACE_INFO1("Clear resolving list status 0x%02x", pMsg->hdr.status);
      break;

    case DM_PER_ADV_SUBEVT_DATA_REQ_IND:
      uint8_t advHandle = pMsg->dm.perAdvDataReq.advHandle;
      uint8_t startSubEvt = pMsg->dm.perAdvDataReq.subevtStart;
      uint8_t subEvtCnt = pMsg->dm.perAdvDataReq.subevtDataCnt;

      hciPerAdvSubEvtDataSetElem_t hciSubEvtDataElem[HCI_LE_SET_SUB_EVT_DATA_NUM_MAX];
      // Specify the same parameters for the subevent
      for ( uint8_t subEvtIdx = startSubEvt; subEvtIdx < subEvtCnt; subEvtIdx++ )
      {
        hciSubEvtDataElem[subEvtIdx].subEvt = subEvtIdx;
        hciSubEvtDataElem[subEvtIdx].rspSlotStart = 0;
        hciSubEvtDataElem[subEvtIdx].rspSlotCnt = NUM_RSP_SLOTS;
        hciSubEvtDataElem[subEvtIdx].subEvtDataLen = SUB_EVT_PACKET_SIZE;

        memcpy(hciSubEvtDataElem[subEvtIdx].subEvtData, &subEvtDataBufs[subEvtIdx*SUB_EVT_PACKET_SIZE], SUB_EVT_PACKET_SIZE);
      }
      APP_TRACE_INFO1("[PER_ADV_DATA][TX] Subevt Cnt: %d", subEvtCnt);
      DmAdvSetPerAdvSbuEvtData(advHandle, subEvtCnt, hciSubEvtDataElem);

      break;

    case DM_HW_ERROR_IND:
      uiEvent = APP_UI_HW_ERROR;
      break;

    case DM_VENDOR_SPEC_CMD_CMPL_IND:
      break;

    default:
      break;
  }

  if (uiEvent != APP_UI_NONE)
  {
    AppUiAction(uiEvent);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Application handler init function called during system initialization.
 *
 *  \param  handlerID  WSF handler ID.
 *
 *  \return None.
 */
/*************************************************************************************************/
void PawrAdvHandlerInit(wsfHandlerId_t handlerId)
{
  APP_TRACE_INFO0("pawrAdvHandlerInit");

  /* store handler ID */
  pawrAdvCb.handlerId = handlerId;

  /* Set configuration pointers */
  pAppExtAdvCfg = (appExtAdvCfg_t*)&pawrAdvAdvCfg;
  pAppSlaveCfg = (appSlaveCfg_t *) &pawrAdvSlaveCfg;
  pAppExtMasterCfg = (appExtMasterCfg_t *) &pawrAdvMasterExtCfg;
  pAppSecCfg = (appSecCfg_t *) &pawrAdvSecCfg;
  /* Set stack configuration pointers */
  pSmpCfg = (smpCfg_t *) &pawrAdvSmpCfg;
  pAttCfg = (attCfg_t *) &pawrAdvAttCfg;
  pAppUpdateCfg = (appUpdateCfg_t *) &pawrAdvUpdateCfg;

  /* Initialize application framework */
  AppSlaveInit();
  AppServerInit();
  AppMasterInit();
}

/*************************************************************************************************/
/*!
 *  \brief  WSF event handler for application.
 *
 *  \param  event   WSF event mask.
 *  \param  pMsg    WSF message.
 *
 *  \return None.
 */
/*************************************************************************************************/
void PawrAdvHandler(wsfEventMask_t event, wsfMsgHdr_t *pMsg)
{
  if (pMsg != NULL)
  {
    /* process ATT messages */
    if (pMsg->event >= ATT_CBACK_START && pMsg->event <= ATT_CBACK_END)
    {
      /* process server-related ATT messages */
      AppServerProcAttMsg(pMsg);
    }
    /* process DM messages */
    else if (pMsg->event >= DM_CBACK_START && pMsg->event <= DM_CBACK_END)
    {
      /* when doing the advertising it works as "slave" role, after connection
       * it works as "master" role.
       */
      if ( AppConnIsOpen() == DM_CONN_ID_NONE )
      {
        /* process advertising and connection-related messages */
        AppSlaveProcDmMsg((dmEvt_t *) pMsg);
      }
      else
      {
        /* process advertising and connection-related messages */
        AppMasterProcDmMsg((dmEvt_t *) pMsg);

        /* process security-related messages */
        AppMasterSecProcDmMsg((dmEvt_t *) pMsg);
      }
    }

    /* perform profile and user interface-related operations */
    pawrAdvProcMsg((pawrAdvMsg_t *) pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Start the application.
 *
 *  \return None.
 */
/*************************************************************************************************/
void PawrAdvStart(void)
{
  /* Register for stack callbacks */
  DmRegister(pawrAdvDmCback);
  DmConnRegister(DM_CLIENT_ID_APP, pawrAdvDmCback);
  AttRegister(pawrAdvAttCback);

  /* Initialize attribute server database */
  SvcCoreAddGroup();

  /* Reset the device */
  DmDevReset();
}
