//*****************************************************************************
//
//! @file pawr_scan_main.c
//!
//! @brief Ambiq Micro's demonstration of Periodic Advertising with Response (Scanner)
//
//*****************************************************************************

//*****************************************************************************
//
// Copyright (c) 2026, Ambiq Micro, Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the copyright holder nor the names of its
// contributors may be used to endorse or promote products derived from this
// software without specific prior written permission.
//
// Third party software included in this distribution is subject to the
// additional license terms as defined in the /docs/licenses directory.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// This is part of revision stable-e24d618f43 of the AmbiqSuite Development Package.
//
//*****************************************************************************

#include <string.h>
#include "wsf_types.h"
#include "util/bstream.h"
#include "wsf_msg.h"
#include "wsf_trace.h"
#include "wsf_assert.h"
#include "wsf_buf.h"
#include "hci_api.h"
#include "dm_api.h"
#include "dm_priv.h"
#include "gap/gap_api.h"
#include "att_api.h"
#include "smp_api.h"
#include "app_cfg.h"
#include "app_api.h"
#include "app_ui.h"
#include "svc_core.h"
#include "svc_ch.h"
#include "gatt/gatt_api.h"
#include "atpc/atpc_api.h"
#include "pawr_scan_api.h"
#include "app_main.h"
#include "am_util_stdio.h"
#include "atts_main.h"
#include "hci_core.h"

/**************************************************************************************************
  Macros
**************************************************************************************************/
#define ADV_BUF_LEN_MAX                 (252 * 5)
#define EXT_SCAN_NAME_MAX_LEN           (253)

/**************************************************************************************************
  Local Variables
**************************************************************************************************/
/*! application control block */
struct
{
  wsfHandlerId_t    handlerId;                      /*! WSF hander ID */
  uint8_t           addrType;                       /*! Scanned advertising address type */
  bdAddr_t          advAddr;                        /*! Scanned advertising address */
  uint8_t           subEvt;                         /*! Subevent number. */
} pawrScanCb;

/**************************************************************************************************
  Configurable Parameters
**************************************************************************************************/
/*! configurable parameters for master extendec scan */
static const appExtMasterCfg_t pawrScanMasterExtCfg =
{
  {96},                                      /*! The scan interval, in 0.625 ms units */
  {48},                                      /*! The scan window, in 0.625 ms units  */
  0,                                         /*! The scan duration in ms */
  0,                                         /*! Scan period*/
  DM_DISC_MODE_GENERAL,                      /*! The GAP discovery mode */
  {DM_SCAN_TYPE_ACTIVE}                      /*! The scan type (active or passive) */
};

/*! configurable parameters for slave */
static const appSlaveCfg_t pawrScanSlaveCfg =
{
  1,                           /*! Maximum connections */
};


/*! ATT configurable parameters (increase MTU) */
static const attCfg_t pawrScanAttCfg =
{
  15,                               /* ATT server service discovery connection idle timeout in seconds */
  23,                               /* desired ATT MTU */
  ATT_MAX_TRANS_TIMEOUT,            /* transcation timeout in seconds */
  4                                 /* number of queued prepare writes supported by server */
};

appExtAdvCfg_t pawrScanCfg =
{
#if (DM_NUM_ADV_SETS == 1)
  {0},
  {800},
  {0},
  {FALSE},
  {0},
  #elif (DM_NUM_ADV_SETS > 1)
  {0, 0},
  {800, 800},
  {0, 0},
  {FALSE, TRUE},
  {0, 0}
#endif
};

/*! configurable parameters for security */
static const appSecCfg_t pawrScanSecCfg =
{
  DM_AUTH_SC_FLAG,      /*! Authentication and bonding flags */
  0,                    /*! Initiator key distribution flags */
  DM_KEY_DIST_LTK,      /*! Responder key distribution flags */
  FALSE,                /*! TRUE if Out-of-band pairing data is present */
  FALSE                 /*! TRUE to initiate security upon connection */
};

/*! configurable parameters for connection parameter update */
static const appUpdateCfg_t pawrScanUpdateCfg =
{
  0,                 /*! Connection idle period in ms before attempting
                            connection parameter update; set to zero to disable */
  48,                   /*! Minimum connection interval in 1.25ms units */
  60,                   /*! Maximum connection interval in 1.25ms units */
  4,                    /*! Connection latency */
  600,                  /*! Supervision timeout in 10ms units */
  5                     /*! Number of update attempts before giving up */
};

/*! SMP security parameter configuration */
static const smpCfg_t pawrScanSmpCfg =
{
  3000,                                   /*! 'Repeated attempts' timeout in msec */
  SMP_IO_NO_IN_NO_OUT,                    /*! I/O Capability */
  7,                                      /*! Minimum encryption key length */
  16,                                     /*! Maximum encryption key length */
  3,                                      /*! Attempts to trigger 'repeated attempts' timeout */
  0,                                      /*! Device authentication requirements */
  64000,                                  /*! Maximum repeated attempts timeout in msec */
  64000,                                  /*! Time msec before attemptExp decreases */
  2                                       /*! Repeated attempts multiplier exponent */
};

/*************************************************************************************************/
/*!
 *  \brief  Application DM callback.
 *
 *  \param  pDmEvt  DM callback event
 *
 *  \return None.
 */
/*************************************************************************************************/
static void pawrScanDmCback(dmEvt_t *pDmEvt)
{
  dmEvt_t   *pMsg;
  uint16_t  len;
  uint16_t  reportLen = 0;

  len = DmSizeOfEvt(pDmEvt);

  if (pDmEvt->hdr.event == DM_EXT_SCAN_REPORT_IND)
  {
    reportLen = pDmEvt->extScanReport.len;
  }
  else if (pDmEvt->hdr.event == DM_PER_ADV_REPORT_IND)
  {
    reportLen = pDmEvt->perAdvReport.len;
  }
  else
  {
    reportLen = 0;
  }

  if ((pMsg = WsfMsgAlloc(len + reportLen)) != NULL)
  {
    memcpy(pMsg, pDmEvt, len);

    if (pDmEvt->hdr.event == DM_EXT_SCAN_REPORT_IND)
    {
      pMsg->extScanReport.pData = (uint8_t *)((uint8_t *)pMsg + len);
      memcpy(pMsg->extScanReport.pData, pDmEvt->extScanReport.pData, reportLen);
    }
    else if (pDmEvt->hdr.event == DM_PER_ADV_REPORT_IND)
    {
      pMsg->perAdvReport.pData = (uint8_t *)((uint8_t *)pMsg + len);
      memcpy(pMsg->perAdvReport.pData, pDmEvt->perAdvReport.pData, reportLen);
    }

    WsfMsgSend(pawrScanCb.handlerId, pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Application  ATT callback.
 *
 *  \param  pEvt    ATT callback event
 *
 *  \return None.
 */
/*************************************************************************************************/
static void pawrScanAttCback(attEvt_t *pEvt)
{
  attEvt_t *pMsg;

  if ((pMsg = WsfMsgAlloc(sizeof(attEvt_t) + pEvt->valueLen)) != NULL)
  {
    memcpy(pMsg, pEvt, sizeof(attEvt_t));
    pMsg->pValue = (uint8_t *) (pMsg + 1);
    memcpy(pMsg->pValue, pEvt->pValue, pEvt->valueLen);
    WsfMsgSend(pawrScanCb.handlerId, pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Handle a extended scan report.
 *
 *  \param  pMsg    Pointer to DM callback event message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void pawrScanExtScanReport(dmEvt_t *pMsg)
{
  uint8_t    *pData;
  char       dev_name[EXT_SCAN_NAME_MAX_LEN + 1] = {0};

  /* find Local name advertising data */
  if ((pData = DmFindAdType(DM_ADV_TYPE_LOCAL_NAME, pMsg->extScanReport.len,
                                 pMsg->extScanReport.pData)) != NULL)
  {
    memcpy(dev_name, (char *)(pData + 2), *pData - 1);
    APP_TRACE_INFO1("Scanned advertising device name: %s", dev_name);

    /* check length and vendor ID */
    if (!strncmp((char *)(pData + 2), "Am_PAwR", *pData - 1))
    {
      APP_TRACE_INFO0("Found expected device");

      uint8_t advSid = pMsg->extScanReport.advSid;
      pawrScanCb.addrType = pMsg->extScanReport.addrType;
      uint16_t skip = 0;             // Do not skip any periodic advertising packets
      uint16_t syncTimeout = 0x03E8; // 1000 * 10ms

      memcpy(pawrScanCb.advAddr, pMsg->extScanReport.addr, BDA_ADDR_LEN);

      AppSyncStart(advSid, pawrScanCb.addrType, pawrScanCb.advAddr, skip, syncTimeout);
      am_util_stdio_printf("Synchronizing with periodic advertising from device: "
        "%02X:%02X:%02X:%02X:%02X:%02X\n",
        pawrScanCb.advAddr[5], pawrScanCb.advAddr[4], pawrScanCb.advAddr[3],
        pawrScanCb.advAddr[2], pawrScanCb.advAddr[1], pawrScanCb.advAddr[0]);
    }
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Process messages from the event handler.
 *
 *  \param  pMsg    Pointer to message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void pawrScanProcMsg(dmEvt_t *pMsg)
{
  uint8_t uiEvent = APP_UI_NONE;

  switch(pMsg->hdr.event)
  {
    case DM_RESET_CMPL_IND:
      // set database hash calculating status to true until a new hash is generated after reset
      attsCsfSetHashUpdateStatus(TRUE);
      AttsCalculateDbHash();
      uiEvent = APP_UI_RESET_CMPL;
      break;

    case DM_SEC_ECC_KEY_IND:
      APP_TRACE_INFO0("ECC key ind");
      break;

    case ATTS_DB_HASH_CALC_CMPL_IND:
      APP_TRACE_INFO0("ATTS_DB_HASH_CALC_CMPL_IND");

      AppExtScanStart(HCI_SCAN_PHY_LE_1M_BIT, pawrScanMasterExtCfg.discMode,
             pawrScanMasterExtCfg.scanType, pawrScanMasterExtCfg.scanDuration, pawrScanMasterExtCfg.scanPeriod);
      break;

    case DM_PER_ADV_REPORT_IND:
    {
      uint16_t syncHandle = pMsg->perAdvReport.syncHandle;
      uint8_t rspDataBuf[9] = {0};
      hciPerAdvRspData_t rspData;
      static uint8_t testCounter = 0;

      if ( AppConnIsOpen() == DM_CONN_ID_NONE )
      {
        rspDataBuf[0] = 0x08;
        rspDataBuf[1] = DM_ADV_TYPE_BD_ADDR;
        memcpy(&rspDataBuf[2], hciCoreCb.bdAddr, sizeof(hciCoreCb.bdAddr));
        rspDataBuf[8] = DM_ADDR_PUBLIC;
        am_util_stdio_printf("[PER_ADV_RSP][TX] Responding with its own BD address: "
          "%02X:%02X:%02X:%02X:%02X:%02X\n",
          rspDataBuf[7], rspDataBuf[6], rspDataBuf[5],
          rspDataBuf[4], rspDataBuf[3], rspDataBuf[2]);
      }
      else
      {
        APP_TRACE_INFO0("[PER_ADV_RSP][TX] Responding with manufacturer data...");
        rspDataBuf[0] = 0x04;
        rspDataBuf[1] = DM_ADV_TYPE_MANUFACTURER;
        rspDataBuf[2] = 0xAC; // Ambiq BT company ID: 0x09AC
        rspDataBuf[3] = 0x09;
        rspDataBuf[4] = testCounter++;
      }

      rspData.syncHandle = syncHandle;
      rspData.requestEvent = pMsg->perAdvReport.evtCnt;
      rspData.requestSubEvt = pMsg->perAdvReport.subEvt;
      rspData.responseSubEvt = pMsg->perAdvReport.subEvt;
      rspData.responseSlot = 0;
      rspData.responseDataLen = rspDataBuf[0] + 1;

      memcpy(rspData.responseData, rspDataBuf, rspData.responseDataLen);

      HciLeSetPerAdvRspDataCmd(&rspData);
      break;
    }

    case DM_EXT_SCAN_REPORT_IND:
      pawrScanExtScanReport(pMsg);
      break;

    case DM_EXT_SCAN_START_IND:
      uiEvent = APP_UI_SCAN_START;
      break;

    case DM_EXT_SCAN_STOP_IND:
      uiEvent = APP_UI_SCAN_STOP;
      break;

    case DM_PER_ADV_SYNC_TRSF_IND:
    case DM_PER_ADV_SYNC_EST_IND:
    {
      uint16_t syncHandle = pMsg->perAdvSyncEst.syncHandle;
      uint16_t prop = 0;
      uint8_t numSubEvt = 1;
      uint8_t subEvt[1];

      AppExtScanStop();

      subEvt[0] = 0;
      DmSyncSetSubEvt(syncHandle, prop, numSubEvt, subEvt);

      break;
    }

    case DM_PER_ADV_SYNC_TRSF_EST_IND:
    {
      uint16_t syncHandle = pMsg->perAdvSyncTrsfEst.syncHandle;
      uint8_t advSid = pMsg->perAdvSyncTrsfEst.advSid;

      APP_TRACE_INFO3("Periodic advertising sync transfer established, "
        "Adv SID: %d, syncHandle:%d, subEvt interval:%d", advSid, syncHandle, pMsg->perAdvSyncEst.subEvtInt);

      uint16_t prop = 0;
      uint8_t numSubEvt = 1;
      uint8_t subEvt[1];
      subEvt[0] = 0;

      DmSyncSetSubEvt(syncHandle, prop, numSubEvt, subEvt);
      DmPastRptRcvEnable(advSid, TRUE);
      break;
    }

    case DM_PER_ADV_SYNC_LOST_IND:
      AppExtScanStart(HCI_SCAN_PHY_LE_1M_BIT, pawrScanMasterExtCfg.discMode,
             pawrScanMasterExtCfg.scanType, pawrScanMasterExtCfg.scanDuration, pawrScanMasterExtCfg.scanPeriod);
      break;

    case DM_CONN_OPEN_IND:
      memcpy(pawrScanCb.advAddr, pMsg->connOpen.peerAddr, 6);
      pawrScanCb.addrType = pMsg->connOpen.addrType;
      uiEvent = APP_UI_CONN_OPEN;
      break;

    case DM_CONN_CLOSE_IND:
      uiEvent = APP_UI_CONN_CLOSE;
      break;

    case DM_PHY_UPDATE_IND:
      APP_TRACE_INFO3("DM_PHY_UPDATE_IND status: %d, RX: %d, TX: %d", pMsg->phyUpdate.status, pMsg->phyUpdate.rxPhy, pMsg->phyUpdate.txPhy);
      break;

    default:
      break;
  }

  if (uiEvent != APP_UI_NONE)
  {
    AppUiAction(uiEvent);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Application handler init function called during system initialization.
 *
 *  \param  handlerID  WSF handler ID.
 *
 *  \return None.
 */
/*************************************************************************************************/
void PawrScanHandlerInit(wsfHandlerId_t handlerId)
{
  APP_TRACE_INFO0("PawrScanHandlerInit");

  /* store handler ID */
  pawrScanCb.handlerId = handlerId;

  /* Set configuration pointers */
  pAppSlaveCfg = (appSlaveCfg_t *) &pawrScanSlaveCfg;
  pAttCfg = (attCfg_t *) &pawrScanAttCfg;

  /* Set configuration pointers */
  pAppExtMasterCfg = (appExtMasterCfg_t *) &pawrScanMasterExtCfg;
  pAppUpdateCfg = (appUpdateCfg_t *) &pawrScanUpdateCfg;
  pAppSecCfg = (appSecCfg_t *) &pawrScanSecCfg;
  pSmpCfg = (smpCfg_t *) &pawrScanSmpCfg;

  /* Initialize application framework */
  AppSlaveInit();
  AppServerInit();

  AppMasterInit();
}

/*************************************************************************************************/
/*!
 *  \brief  WSF event handler for application.
 *
 *  \param  event   WSF event mask.
 *  \param  pMsg    WSF message.
 *
 *  \return None.
 */
/*************************************************************************************************/
void PawrScanHandler(wsfEventMask_t event, wsfMsgHdr_t *pMsg)
{
  if (pMsg != NULL)
  {
    /* process ATT messages */
    if (pMsg->event >= ATT_CBACK_START && pMsg->event <= ATT_CBACK_END)
    {
      /* process server-related ATT messages */
      AppServerProcAttMsg(pMsg);
    }
    /* process DM messages */
    else if (pMsg->event >= DM_CBACK_START && pMsg->event <= DM_CBACK_END)
    {
      /* when doing the scanning it works as "master" role, after connection
       * it works as "slave" role.
       */
      if ( pMsg->param == DM_CONN_ID_NONE )
      {
        /* process advertising and connection-related messages */
        AppMasterProcDmMsg((dmEvt_t *) pMsg);
      }
      else
      {
        /* process advertising and connection-related messages */
        AppSlaveProcDmMsg((dmEvt_t *) pMsg);

        /* process security-related messages */
        AppSlaveSecProcDmMsg((dmEvt_t *) pMsg);
      }
    }

    /* perform profile and user interface-related operations */
    pawrScanProcMsg((dmEvt_t *) pMsg);
  }
}


/*************************************************************************************************/
/*!
 *  \brief  Start the application.
 *
 *  \return None.
 */
/*************************************************************************************************/
void PawrScanStart(void)
{
  /* Register for stack callbacks */
  DmRegister(pawrScanDmCback);
  DmConnRegister(DM_CLIENT_ID_APP, pawrScanDmCback);
  AttRegister(pawrScanAttCback);
  AttConnRegister(AppServerConnCback);

  /* Initialize attribute server database */
  SvcCoreGattCbackRegister(GattReadCback, GattWriteCback);
  SvcCoreAddGroup();

  /* Reset the device */
  DmDevReset();
}
