//*****************************************************************************
//
//! @file periodic_adv_main.c
//!
//! @brief Ambiq Micro's demonstration of Periodic Advertising.
//
//*****************************************************************************

//*****************************************************************************
//
// Copyright (c) 2026, Ambiq Micro, Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the copyright holder nor the names of its
// contributors may be used to endorse or promote products derived from this
// software without specific prior written permission.
//
// Third party software included in this distribution is subject to the
// additional license terms as defined in the /docs/licenses directory.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// This is part of revision stable-e24d618f43 of the AmbiqSuite Development Package.
//
//*****************************************************************************

#include <string.h>
#include "wsf_types.h"
#include "util/bstream.h"
#include "wsf_msg.h"
#include "wsf_trace.h"
#include "hci_api.h"
#include "dm_api.h"
#include "att_api.h"
#include "smp_api.h"
#include "app_api.h"
#include "app_db.h"
#include "app_ui.h"
#include "app_hw.h"
#include "app_main.h"
#include "svc_ch.h"
#include "svc_core.h"
#include "svc_dis.h"
#include "gatt/gatt_api.h"
#include "periodic_adv_api.h"
#include "atts_main.h"
/**************************************************************************************************
  Macros
**************************************************************************************************/
#if defined(BLE_PER_ADV_DATA_DID)
#define PER_ADV_DID_CHG_TIMER_SEC  5
wsfTimer_t     perAdvDIDChangeTimer;
#endif
/*! WSF message event starting value */
#define PERIODIC_ADV_MSG_START           0xA0
#define ADV_BUF_LEN_MAX                 (252*5)

/*! WSF message event enumeration */
enum
{
  PERIODIC_ADV_FIRST_TIMER_IND = PERIODIC_ADV_MSG_START,
#if (BT_53)
#if defined(BLE_PER_ADV_DATA_DID)
  ADV_EXT_PRE_ADV_DID_TIMER_IND,                            /* Periodic avdertising DID change timer */
#endif // defined(BLE_PER_ADV_DATA_DID)
#endif // BT_53
};

/**************************************************************************************************
  Data Types
**************************************************************************************************/

/*! Application message type */
typedef union
{
  wsfMsgHdr_t     hdr;
  dmEvt_t         dm;
  attsCccEvt_t    ccc;
  attEvt_t        att;
} periodicAdvMsg_t;

/**************************************************************************************************
  Configurable Parameters
**************************************************************************************************/

appExtAdvCfg_t periodicAdvAdvCfg =
{
#if (DM_NUM_ADV_SETS == 1)
    {0},
    {800},
    {0},
    {FALSE},
    {0},
#elif (DM_NUM_ADV_SETS > 1)
    {0, 0},
    {800, 800},
    {0, 0},
    {FALSE, TRUE},
    {800, 800},
#endif
};

static uint8_t adv_handle[DM_NUM_ADV_SETS] = {0};
extern appExtConnCb_t appExtConnCb[DM_CONN_MAX];

/*! advertising data, discoverable mode */
static const uint8_t periodicAdvAdvDataDisc_per[] =
{
  /*! flags */
  2,                                      /*! length */
  DM_ADV_TYPE_FLAGS,                      /*! AD type */
  DM_FLAG_LE_GENERAL_DISC |               /*! flags */
  DM_FLAG_LE_BREDR_NOT_SUP,

  /*! device name */
  11,                                     /*! length */
  DM_ADV_TYPE_LOCAL_NAME,                 /*! AD type */
  'A',
  'm',
  '_',
  'P',
  'e',
  'r',
  '_',
  'A',
  'd',
  'v'
};

/*! configurable parameters for slave */
static const appSlaveCfg_t periodicAdvSlaveCfg =
{
  PERIODIC_ADV_CONN_MAX,                  /*! Maximum connections */
};

/*! configurable parameters for security */
static const appSecCfg_t periodicAdvSecCfg =
{
  DM_AUTH_SC_FLAG,                        /*! Authentication and bonding flags */
  0,                                      /*! Initiator key distribution flags */
  DM_KEY_DIST_LTK,                        /*! Responder key distribution flags */
  FALSE,                                  /*! TRUE if Out-of-band pairing data is present */
  FALSE                                   /*! TRUE to initiate security upon connection */
};

/*! configurable parameters for connection parameter update */
static const appUpdateCfg_t periodicAdvUpdateCfg =
{
  3000,                                   /*! Connection idle period in ms before attempting
                                              connection parameter update; set to zero to disable */
  48,                                     /*! Minimum connection interval in 1.25ms units */
  60,                                     /*! Maximum connection interval in 1.25ms units */
  4,                                      /*! Connection latency */
  600,                                    /*! Supervision timeout in 10ms units */
  5                                       /*! Number of update attempts before giving up */
};

/*! SMP security parameter configuration */
static const smpCfg_t periodicAdvSmpCfg =
{
  3000,                                   /*! 'Repeated attempts' timeout in msec */
  SMP_IO_NO_IN_NO_OUT,                    /*! I/O Capability */
  7,                                      /*! Minimum encryption key length */
  16,                                     /*! Maximum encryption key length */
  3,                                      /*! Attempts to trigger 'repeated attempts' timeout */
  0,                                      /*! Device authentication requirements */
  64000,                                  /*! Maximum repeated attempts timeout in msec */
  64000,                                  /*! Time msec before attemptExp decreases */
  2                                       /*! Repeated attempts multiplier exponent */
};

/**************************************************************************************************
  Advertising Data
**************************************************************************************************/
static char perAdvName[] = "Periodic_Adv_Test";

/**************************************************************************************************
  Client Characteristic Configuration Descriptors
**************************************************************************************************/

/*! enumeration of client characteristic configuration descriptors */
enum
{
  PERIODIC_ADV_GATT_SC_CCC_IDX,                    /*! GATT service, service changed characteristic */
  PERIODIC_ADV_NUM_CCC_IDX
};

/*! client characteristic configuration descriptors settings, indexed by above enumeration */
static const attsCccSet_t periodicAdvCccSet[PERIODIC_ADV_NUM_CCC_IDX] =
{
  /* cccd handle          value range               security level */
  {GATT_SC_CH_CCC_HDL,    ATT_CLIENT_CFG_INDICATE,  DM_SEC_LEVEL_NONE},   /* PERIODIC_ADV_GATT_SC_CCC_IDX */
};

/**************************************************************************************************
  Variables
**************************************************************************************************/

/*! WSF handler ID */
wsfHandlerId_t periodicAdvHandlerId;

typedef struct
{
  hciExtAdvParam_t advParam;
  uint8_t advType;
  bool_t useLegAdvPdu;
}extAdvParam_t;

static uint8_t peerAddr[BDA_ADDR_LEN] = {0x12, 0x34, 0x56, 0x78, 0xAB, 0xFF};
static uint8_t perDataBuf[ADV_BUF_LEN_MAX] = {0};

static extAdvParam_t extAdvParam[DM_NUM_ADV_SETS] =
{
  {
    {
      .priAdvInterMin  = 1000 / 0.625,      // 1s minimum interval
      .priAdvInterMax  = 1000 / 0.625,
      .priAdvChanMap   = DM_ADV_CHAN_ALL,
      .ownAddrType     = DM_ADDR_PUBLIC,
      .peerAddrType    = DM_ADDR_PUBLIC,
      .pPeerAddr       = peerAddr,
      .advFiltPolicy   = HCI_ADV_FILT_NONE,
      .advTxPwr        = HCI_TX_PWR_NO_PREFERENCE,
      .priAdvPhy       = HCI_ADV_PHY_LE_1M,
      .secAdvMaxSkip   = 0,
      .secAdvPhy       = HCI_ADV_PHY_LE_1M,
      .advSID          = 0,
      .scanReqNotifEna = FALSE,
    },
    .advType           = DM_ADV_CONN_UNDIRECT,
    .useLegAdvPdu      = FALSE,

  },

  {
      {
        .priAdvInterMin  = 1000 / 0.625,    // 1s minimum interval
        .priAdvInterMax  = 1000 / 0.625,
        .priAdvChanMap   = DM_ADV_CHAN_ALL,
        .ownAddrType     = DM_ADDR_PUBLIC,
        .peerAddrType    = DM_ADDR_PUBLIC,
        .pPeerAddr       = peerAddr,
        .advFiltPolicy   = HCI_ADV_FILT_NONE,
        .advTxPwr        = HCI_TX_PWR_NO_PREFERENCE,
        .priAdvPhy       = HCI_ADV_PHY_LE_1M,
        .secAdvMaxSkip   = 0,
        .secAdvPhy       = HCI_ADV_PHY_LE_2M,
        .advSID          = 1,
        .scanReqNotifEna = FALSE,
      },
      .advType           = DM_EXT_ADV_CONN_UNDIRECT,    //DM_ADV_CONN_UNDIRECT,
      .useLegAdvPdu      = FALSE,
  }
};


/*************************************************************************************************/
/*!
 *  \brief  Application DM callback.
 *
 *  \param  pDmEvt  DM callback event
 *
 *  \return None.
 */
/*************************************************************************************************/
static void periodicAdvDmCback(dmEvt_t *pDmEvt)
{
  dmEvt_t *pMsg;
  uint16_t len;

  len = DmSizeOfEvt(pDmEvt);

  if ((pMsg = WsfMsgAlloc(len)) != NULL)
  {
    memcpy((uint8_t *)pMsg, (uint8_t *)pDmEvt, len);
    WsfMsgSend(periodicAdvHandlerId, pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Application ATT callback.
 *
 *  \param  pEvt    ATT callback event
 *
 *  \return None.
 */
/*************************************************************************************************/
static void periodicAdvAttCback(attEvt_t *pEvt)
{
  attEvt_t *pMsg;

  if ((pMsg = WsfMsgAlloc(sizeof(attEvt_t) + pEvt->valueLen)) != NULL)
  {
    memcpy((uint8_t *)pMsg, (uint8_t *)pEvt, sizeof(attEvt_t));
    pMsg->pValue = (uint8_t *) (pMsg + 1);
    memcpy(pMsg->pValue, pEvt->pValue, pEvt->valueLen);
    WsfMsgSend(periodicAdvHandlerId, pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Application ATTS client characteristic configuration callback.
 *
 *  \param  pDmEvt  DM callback event
 *
 *  \return None.
 */
/*************************************************************************************************/
static void periodicAdvCccCback(attsCccEvt_t *pEvt)
{
  attsCccEvt_t  *pMsg;
  appDbHdl_t    dbHdl;

  /* If CCC not set from initialization and there's a device record and currently bonded */
  if ((pEvt->handle != ATT_HANDLE_NONE) &&
      ((dbHdl = AppDbGetHdl((dmConnId_t) pEvt->hdr.param)) != APP_DB_HDL_NONE) &&
      AppCheckBonded((dmConnId_t)pEvt->hdr.param))
  {
    /* Store value in device database. */
    AppDbSetCccTblValue(dbHdl, pEvt->idx, pEvt->value);
  }

  if ((pMsg = WsfMsgAlloc(sizeof(attsCccEvt_t))) != NULL)
  {
    memcpy((uint8_t *)pMsg, (uint8_t *)pEvt, sizeof(attsCccEvt_t));
    WsfMsgSend(periodicAdvHandlerId, pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Process CCC state change.
 *
 *  \param  pMsg    Pointer to message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void periodicAdvProcCccState(periodicAdvMsg_t *pMsg)
{
  APP_TRACE_INFO3("ccc state ind value:%d handle:%d idx:%d", pMsg->ccc.value, pMsg->ccc.handle, pMsg->ccc.idx);
}

/*************************************************************************************************/
/*!
 *  \brief  Perform UI actions on connection close.
 *
 *  \param  pMsg    Pointer to message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void periodicAdvClose(periodicAdvMsg_t *pMsg)
{
  APP_TRACE_INFO0("Periodic Adv Close");
}


/*************************************************************************************************/
/*!
 *  \brief  Set up advertising parameters used in Extended Advertising
 *
 *  \param  NULL.
 *
 *  \return None.
 */
/*************************************************************************************************/
void AppExtSetParam(void)
{
  uint8_t advHandle = 0;

  for ( advHandle = 0; advHandle < DM_NUM_ADV_SETS; advHandle++ )
  {
    periodicAdvAdvCfg.advInterval[advHandle] = (uint16_t)(extAdvParam[advHandle].advParam.priAdvInterMin);

    DmAdvSetChannelMap(advHandle, extAdvParam[advHandle].advParam.priAdvChanMap);

    DmAdvSetAddrType(extAdvParam[advHandle].advParam.ownAddrType);

    AppExtSetAdvPeerAddr(advHandle, extAdvParam[advHandle].advParam.peerAddrType,
                        extAdvParam[advHandle].advParam.pPeerAddr);

    DmDevSetExtFilterPolicy(advHandle, DM_FILT_POLICY_MODE_ADV, HCI_ADV_FILT_NONE);

    DmAdvIncTxPwr(advHandle, FALSE, extAdvParam[advHandle].advParam.advTxPwr);

    DmAdvScanReqNotifEnable(advHandle, extAdvParam[advHandle].advParam.scanReqNotifEna);

    AppExtSetAdvType(advHandle, extAdvParam[advHandle].advType);

    DmAdvUseLegacyPdu(advHandle, extAdvParam[advHandle].useLegAdvPdu);
    periodicAdvAdvCfg.useLegacyPdu[advHandle] = extAdvParam[advHandle].useLegAdvPdu;
#if (BT_53)
#if defined(BLE_PER_ADI)
    DmPerAdvIncAdi(advHandle, TRUE);
#endif // defined(BLE_PER_ADI)
#endif // BT_53
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Set up advertising and other procedures that need to be performed after
 *          device reset.
 *
 *  \param  pMsg    Pointer to message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void periodicAdvSetup(periodicAdvMsg_t *pMsg)
{
  APP_TRACE_INFO0("Starting Periodic advertising");

  AppExtAdvSetData(adv_handle[0], APP_ADV_DATA_DISCOVERABLE, sizeof(periodicAdvAdvDataDisc_per),
                    (uint8_t *)periodicAdvAdvDataDisc_per, ADV_BUF_LEN_MAX);
  AppExtSetParam();
  AppExtSetAdvType(adv_handle[0], DM_ADV_NONCONN_UNDIRECT);
  AppExtAdvStart(1, &adv_handle[0], APP_MODE_AUTO_INIT);

  memset(perDataBuf, 0x0, sizeof(perDataBuf));
  AppPerAdvSetData(adv_handle[0], 0, (uint8_t *) perDataBuf, ADV_BUF_LEN_MAX);

  AppPerAdvSetAdValue(adv_handle[0], DM_ADV_TYPE_LOCAL_NAME, strlen(perAdvName), (uint8_t *) perAdvName);

  AppPerAdvStart(adv_handle[0]);
}

/*************************************************************************************************/
/*!
 *  \brief  Process messages from the event handler.
 *
 *  \param  pMsg    Pointer to message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void periodicAdvProcMsg(periodicAdvMsg_t *pMsg)
{
  uint8_t uiEvent = APP_UI_NONE;
  uint8_t i = 0;

  switch(pMsg->hdr.event)
  {
#if (BT_53)
#if defined(BLE_PER_ADV_DATA_DID)
    case ADV_EXT_PRE_ADV_DID_TIMER_IND:
      AppUpdatePerAdvDid(adv_handle[0]);
      WsfTimerStartSec(&perAdvDIDChangeTimer, PER_ADV_DID_CHG_TIMER_SEC);
      break;
#endif // defined(BLE_PER_ADV_DATA_DID)
#endif // BT_53

    case ATTS_HANDLE_VALUE_CNF:
      break;

    case ATTS_CCC_STATE_IND:
      periodicAdvProcCccState(pMsg);
      break;

    case ATT_MTU_UPDATE_IND:
      APP_TRACE_INFO1("Negotiated MTU %d", ((attEvt_t *)pMsg)->mtu);
      break;

    case DM_RESET_CMPL_IND:
      // set database hash calculating status to true until a new hash is generated after reset
      attsCsfSetHashUpdateStatus(TRUE);

      // Generate ECC key if configured support secure connection,
      // else will calcualte ATT database hash
      if ( periodicAdvSecCfg.auth & DM_AUTH_SC_FLAG )
      {
          DmSecGenerateEccKeyReq();
      }
      else
      {
          AttsCalculateDbHash();
      }

      uiEvent = APP_UI_RESET_CMPL;
      break;

    case ATTS_DB_HASH_CALC_CMPL_IND:
      periodicAdvSetup(pMsg);
      break;

    case DM_ADV_SET_START_IND:
      uiEvent = APP_UI_ADV_SET_START_IND;
#if (BT_53)
#if defined(BLE_PER_ADV_DATA_DID)
      WsfTimerStartSec(&perAdvDIDChangeTimer, PER_ADV_DID_CHG_TIMER_SEC);
#endif // defined(BLE_PER_ADV_DATA_DID)
#endif // BT_53
      break;

    case DM_ADV_SET_STOP_IND:
    {
        for ( i = 0; i < DM_CONN_MAX; i++ )
        {
          if ( !appExtConnCb[i].used )
          {
              appExtConnCb[i].used = TRUE;
              appExtConnCb[i].advHandle = pMsg->dm.advSetStop.advHandle;
              appExtConnCb[i].connHandle = pMsg->dm.advSetStop.handle;

              break;
          }
        }

        uiEvent = APP_UI_ADV_SET_STOP_IND;
    }
    break;

    case DM_ADV_START_IND:
        uiEvent = APP_UI_ADV_START;
    break;

    case DM_ADV_STOP_IND:
      uiEvent = APP_UI_ADV_STOP;
      break;

    case DM_CONN_OPEN_IND:
      uiEvent = APP_UI_CONN_OPEN;
      break;

    case DM_CONN_CLOSE_IND:
    {
      uint8_t connHdl = pMsg->dm.connClose.handle;

      for ( i = 0; i < DM_CONN_MAX; i++ )
      {
        if ( appExtConnCb[i].used && (connHdl == appExtConnCb[i].connHandle) )
        {
          appExtConnCb[i].used = FALSE;
          AppExtAdvStart(1, &appExtConnCb[i].advHandle, APP_MODE_AUTO_INIT);

          break;
        }
      }

      periodicAdvClose(pMsg);
      uiEvent = APP_UI_CONN_CLOSE;
    }
      break;

    case DM_PHY_UPDATE_IND:
      APP_TRACE_INFO3("DM_PHY_UPDATE_IND status: %d, RX: %d, TX: %d", pMsg->dm.phyUpdate.status, pMsg->dm.phyUpdate.rxPhy, pMsg->dm.phyUpdate.txPhy);
      break;

    case DM_SEC_PAIR_CMPL_IND:
      DmSecGenerateEccKeyReq();
      uiEvent = APP_UI_SEC_PAIR_CMPL;
      break;

    case DM_SEC_PAIR_FAIL_IND:
      DmSecGenerateEccKeyReq();
      uiEvent = APP_UI_SEC_PAIR_FAIL;
      break;

    case DM_SEC_ENCRYPT_IND:
      uiEvent = APP_UI_SEC_ENCRYPT;
      break;

    case DM_SEC_ENCRYPT_FAIL_IND:
      uiEvent = APP_UI_SEC_ENCRYPT_FAIL;
      break;

    case DM_SEC_AUTH_REQ_IND:
      AppHandlePasskey(&pMsg->dm.authReq);
      break;

    case DM_SEC_ECC_KEY_IND:
      DmSecSetEccKey(&pMsg->dm.eccMsg.data.key);
      // Only calculate database hash if the calculating status is in progress
      if ( attsCsfGetHashUpdateStatus() )
      {
        AttsCalculateDbHash();
      }
      break;

    case DM_SEC_COMPARE_IND:
      AppHandleNumericComparison(&pMsg->dm.cnfInd);
      break;

    case DM_PRIV_CLEAR_RES_LIST_IND:
      APP_TRACE_INFO1("Clear resolving list status 0x%02x", pMsg->hdr.status);
      break;

    case DM_HW_ERROR_IND:
      uiEvent = APP_UI_HW_ERROR;
      break;

    case DM_VENDOR_SPEC_CMD_CMPL_IND:
      break;

    default:
      break;
  }

  if (uiEvent != APP_UI_NONE)
  {
    AppUiAction(uiEvent);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Application handler init function called during system initialization.
 *
 *  \param  handlerID  WSF handler ID.
 *
 *  \return None.
 */
/*************************************************************************************************/
void PeriodicAdvHandlerInit(wsfHandlerId_t handlerId)
{
  APP_TRACE_INFO0("periodicAdvHandlerInit");

  /* store handler ID */
  periodicAdvHandlerId = handlerId;

  /* Set configuration pointers */
  pAppExtAdvCfg = (appExtAdvCfg_t*)&periodicAdvAdvCfg;
  pAppSlaveCfg = (appSlaveCfg_t *) &periodicAdvSlaveCfg;
  pAppSecCfg = (appSecCfg_t *) &periodicAdvSecCfg;
  pAppUpdateCfg = (appUpdateCfg_t *) &periodicAdvUpdateCfg;

  /* Initialize application framework */
  AppSlaveInit();
  AppServerInit();

  /* Set stack configuration pointers */
  pSmpCfg = (smpCfg_t *) &periodicAdvSmpCfg;

#if (BT_53)
#if defined(BLE_PER_ADV_DATA_DID)
  perAdvDIDChangeTimer.msg.event = ADV_EXT_PRE_ADV_DID_TIMER_IND;
  perAdvDIDChangeTimer.handlerId = handlerId;
#endif // defined(BLE_PER_ADV_DATA_DID)
#endif // BT_53
}

/*************************************************************************************************/
/*!
 *  \brief  WSF event handler for application.
 *
 *  \param  event   WSF event mask.
 *  \param  pMsg    WSF message.
 *
 *  \return None.
 */
/*************************************************************************************************/
void PeriodicAdvHandler(wsfEventMask_t event, wsfMsgHdr_t *pMsg)
{
  if (pMsg != NULL)
  {
    APP_TRACE_INFO1("periodicAdv got evt %d", pMsg->event);

    /* process ATT messages */
    if (pMsg->event >= ATT_CBACK_START && pMsg->event <= ATT_CBACK_END)
    {
      /* process server-related ATT messages */
      AppServerProcAttMsg(pMsg);
    }
    /* process DM messages */
    else if (pMsg->event >= DM_CBACK_START && pMsg->event <= DM_CBACK_END)
    {
      /* process advertising and connection-related messages */
      AppSlaveProcDmMsg((dmEvt_t *) pMsg);

      /* process security-related messages */
      AppSlaveSecProcDmMsg((dmEvt_t *) pMsg);
    }

    /* perform profile and user interface-related operations */
    periodicAdvProcMsg((periodicAdvMsg_t *) pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Start the application.
 *
 *  \return None.
 */
/*************************************************************************************************/
void PeriodicAdvStart(void)
{
  /* Register for stack callbacks */
  DmRegister(periodicAdvDmCback);
  DmConnRegister(DM_CLIENT_ID_APP, periodicAdvDmCback);
  AttRegister(periodicAdvAttCback);
  AttConnRegister(AppServerConnCback);
  AttsCccRegister(PERIODIC_ADV_NUM_CCC_IDX, (attsCccSet_t *) periodicAdvCccSet, periodicAdvCccCback);

  /* Initialize attribute server database */
  SvcCoreGattCbackRegister(GattReadCback, GattWriteCback);
  SvcCoreAddGroup();

  /* Set Service Changed CCCD index. */
  GattSetSvcChangedIdx(PERIODIC_ADV_GATT_SC_CCC_IDX);

  /* Reset the device */
  DmDevReset();
}
