// ****************************************************************************
//
//  peripheral_past_main.c
//! @file
//!
//! @brief Peripheral PAST sample application.
//!
//!
//! @{
//
// ****************************************************************************

//*****************************************************************************
//
// Copyright (c) 2026, Ambiq Micro, Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the copyright holder nor the names of its
// contributors may be used to endorse or promote products derived from this
// software without specific prior written permission.
//
// Third party software included in this distribution is subject to the
// additional license terms as defined in the /docs/licenses directory.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// This is part of revision stable-e24d618f43 of the AmbiqSuite Development Package.
//
//*****************************************************************************

#include <string.h>
#include "wsf_types.h"
#include "util/bstream.h"
#include "wsf_msg.h"
#include "wsf_trace.h"
#include "hci_api.h"
#include "dm_api.h"
#include "att_api.h"
#include "smp_api.h"
#include "app_api.h"
#include "app_db.h"
#include "app_ui.h"
#include "app_hw.h"
#include "app_main.h"
#include "svc_ch.h"
#include "svc_core.h"
#include "svc_hrs.h"
#include "svc_dis.h"
#include "svc_batt.h"
#include "svc_rscs.h"
#include "gatt/gatt_api.h"
#include "bas/bas_api.h"
#include "hrps/hrps_api.h"
#include "rscp/rscp_api.h"
#include "peripheral_past_api.h"
#include "atts_main.h"
/**************************************************************************************************
  Macros
**************************************************************************************************/
/*! WSF message event starting value */
#define PERIPHERAL_PAST_MSG_START               0xA0
#define ADV_BUF_LEN_MAX                 (252*5)

/* Default Running Speed and Cadence Measurement period (seconds) */
#define PERIPHERAL_PAST_DEFAULT_RSCM_PERIOD        1

/*! WSF message event enumeration */
enum
{
  PERIPHERAL_PAST_HR_TIMER_IND = PERIPHERAL_PAST_MSG_START,       /*! Heart rate measurement timer expired */
  PERIPHERAL_PAST_BATT_TIMER_IND,                     /*! Battery measurement timer expired */
  PERIPHERAL_PAST_RUNNING_TIMER_IND                   /*! Running speed and cadence measurement timer expired */
};

/**************************************************************************************************
  Data Types
**************************************************************************************************/

/*! Application message type */
typedef union
{
  wsfMsgHdr_t     hdr;
  dmEvt_t         dm;
  attsCccEvt_t    ccc;
  attEvt_t        att;
} peripheralPastMsg_t;

/**************************************************************************************************
  Configurable Parameters
**************************************************************************************************/

appExtAdvCfg_t peripheralPastAdvCfg =
{
#if (DM_NUM_ADV_SETS == 1)
    {0},
    {800},
    {0},
    {FALSE},
    {0},
#elif (DM_NUM_ADV_SETS > 1)
    {0, 0},
    {800, 800},
    {0, 0},
    {FALSE, TRUE},
    {0, 0}
#endif
};

uint8_t adv_handle[DM_NUM_ADV_SETS] = {0};
extern appExtConnCb_t appExtConnCb[DM_CONN_MAX];

/*! configurable parameters for slave */
static const appSlaveCfg_t peripheralPastSlaveCfg =
{
  PERIPHERAL_PAST_CONN_MAX,                           /*! Maximum connections */
};

/*! configurable parameters for security */
static const appSecCfg_t peripheralPastSecCfg =
{
  DM_AUTH_SC_FLAG,    /*! Authentication and bonding flags */
  0,                                      /*! Initiator key distribution flags */
  DM_KEY_DIST_LTK,                        /*! Responder key distribution flags */
  FALSE,                                  /*! TRUE if Out-of-band pairing data is present */
  FALSE                                    /*! TRUE to initiate security upon connection */
};

/*! configurable parameters for connection parameter update */
static const appUpdateCfg_t peripheralPastUpdateCfg =
{
  3000,                                      /*! Connection idle period in ms before attempting
                                              connection parameter update; set to zero to disable */
  48,                                    /*! Minimum connection interval in 1.25ms units */
  60,                                    /*! Maximum connection interval in 1.25ms units */
  4,                                      /*! Connection latency */
  600,                                    /*! Supervision timeout in 10ms units */
  5                                       /*! Number of update attempts before giving up */
};

/*! heart rate measurement configuration */
static const hrpsCfg_t peripheralPastHrpsCfg =
{
  100      /*! Measurement timer expiration period in ms */
};

/*! battery measurement configuration */
static const basCfg_t peripheralPastBasCfg =
{
  30,       /*! Battery measurement timer expiration period in seconds */
  1,        /*! Perform battery measurement after this many timer periods */
  100       /*! Send battery level notification to peer when below this level. */
};

/*! SMP security parameter configuration */
static const smpCfg_t peripheralPastSmpCfg =
{
  3000,                                   /*! 'Repeated attempts' timeout in msec */
  SMP_IO_NO_IN_NO_OUT,                    /*! I/O Capability */
  7,                                      /*! Minimum encryption key length */
  16,                                     /*! Maximum encryption key length */
  3,                                      /*! Attempts to trigger 'repeated attempts' timeout */
  0,                                      /*! Device authentication requirements */
  64000,                                  /*! Maximum repeated attempts timeout in msec */
  64000,                                  /*! Time msec before attemptExp decreases */
  2                                       /*! Repeated attempts multiplier exponent */
};

/**************************************************************************************************
  Advertising Data
**************************************************************************************************/
/*! advertising data, discoverable mode */
static const uint8_t peripheralPastAdvDataDisc[] =
{
  /*! flags */
  2,                                      /*! length */
  DM_ADV_TYPE_FLAGS,                      /*! AD type */
  DM_FLAG_LE_GENERAL_DISC |               /*! flags */
  DM_FLAG_LE_BREDR_NOT_SUP,

  /*! tx power */
  2,                                      /*! length */
  DM_ADV_TYPE_TX_POWER,                   /*! AD type */
  0,                                      /*! tx power */

  /*! service UUID list */
  9,                                      /*! length */
  DM_ADV_TYPE_16_UUID,                    /*! AD type */
  UINT16_TO_BYTES(ATT_UUID_HEART_RATE_SERVICE),
  UINT16_TO_BYTES(ATT_UUID_RUNNING_SPEED_SERVICE),
  UINT16_TO_BYTES(ATT_UUID_DEVICE_INFO_SERVICE),
  UINT16_TO_BYTES(ATT_UUID_BATTERY_SERVICE),

    /*! device name */
  8,                                      /*! length */
  DM_ADV_TYPE_LOCAL_NAME,                 /*! AD type */
  'A',
  'm',
  'b',
  'q',
  '_',
  'A',
  'E'
};

/*! scan data, discoverable mode */
static const uint8_t peripheralPastScanDataDisc[] =
{
  /*! device name */
  12,                                      /*! length */
  DM_ADV_TYPE_LOCAL_NAME,                 /*! AD type */
  'A',
  'm',
  'b',
  'i',
  'q',
  '_',
  'A',
  'd',
  'E',
  'x',
  't'
};

/**************************************************************************************************
  Client Characteristic Configuration Descriptors
**************************************************************************************************/

/*! enumeration of client characteristic configuration descriptors */
enum
{
  PERIPHERAL_PAST_GATT_SC_CCC_IDX,                    /*! GATT service, service changed characteristic */
  PERIPHERAL_PAST_HRS_HRM_CCC_IDX,                    /*! Heart rate service, heart rate monitor characteristic */
  PERIPHERAL_PAST_BATT_LVL_CCC_IDX,                   /*! Battery service, battery level characteristic */
  PERIPHERAL_PAST_RSCS_SM_CCC_IDX,                   /*! Runninc speed and cadence measurement characteristic */
  PERIPHERAL_PAST_NUM_CCC_IDX
};

/*! client characteristic configuration descriptors settings, indexed by above enumeration */
static const attsCccSet_t peripheralPastCccSet[PERIPHERAL_PAST_NUM_CCC_IDX] =
{
  /* cccd handle          value range               security level */
  {GATT_SC_CH_CCC_HDL,    ATT_CLIENT_CFG_INDICATE,  DM_SEC_LEVEL_NONE},   /* PERIPHERAL_PAST_GATT_SC_CCC_IDX */
  {HRS_HRM_CH_CCC_HDL,    ATT_CLIENT_CFG_NOTIFY,    DM_SEC_LEVEL_NONE},   /* PERIPHERAL_PAST_HRS_HRM_CCC_IDX */
  {BATT_LVL_CH_CCC_HDL,   ATT_CLIENT_CFG_NOTIFY,    DM_SEC_LEVEL_NONE},   /* PERIPHERAL_PAST_BATT_LVL_CCC_IDX */
  {RSCS_RSM_CH_CCC_HDL,   ATT_CLIENT_CFG_NOTIFY,    DM_SEC_LEVEL_NONE}    /* PERIPHERAL_PAST_RSCS_SM_CCC_IDX */
};

/**************************************************************************************************
  Global Variables
**************************************************************************************************/

/*! WSF handler ID */
wsfHandlerId_t peripheralPastHandlerId;

/* WSF Timer to send running speed and cadence measurement data */
wsfTimer_t     peripheralPastRscmTimer;

/* Running Speed and Cadence Measurement period - Can be changed at runtime to vary period */
static uint16_t peripheralPastRscmPeriod = PERIPHERAL_PAST_DEFAULT_RSCM_PERIOD;

/* Heart Rate Monitor feature flags */
static uint8_t peripheralPastHrmFlags = CH_HRM_FLAGS_VALUE_8BIT | CH_HRM_FLAGS_ENERGY_EXP;

typedef enum
{
    MODE_TYPE_INVALID,
    MODE_TYPE_PRIM_PHY,
    MODE_TYPE_SEC_PHY,
    MODE_TYPE_ADV_TYPE,
    MODE_TYPE_MAX
}advModeSel_t;

uint8_t advPrimPhyChanl  = HCI_ADV_PHY_LE_1M;
uint8_t advSecdAdvChanl  = HCI_ADV_PHY_LE_1M;
advModeSel_t modeSet     = MODE_TYPE_INVALID;
uint8_t advEvtType       = DM_ADV_NONE;

char *modeStr[] =
{
    NULL,
    "Set primary PHY",
    "Set Secondary PHY",
    "Set Adv Event Type"
};

char *advChannelStr[] =
{
    "LE 1M",
    "LE 2M",
    "LE Coded"
};

char *advTypeStr[] =
{
    "ADV_CONN_UNDIRECT",
    "ADV_CONN_DIRECT",
    "ADV_SCAN_UNDIRECT",
    "ADV_NONCONN_UNDIRECT",

    "ADV_CONN_DIRECT_LO_DUTY",
    "EXT_ADV_CONN_UNDIRECT",
    "EXT_ADV_NONCONN_DIRECT",
    "EXT_ADV_SCAN_DIRECT",
};

typedef struct
{
    hciExtAdvParam_t advParam;
    uint8_t advType;
    bool_t useLegAdvPdu;
}extAdvParam_t;

uint8_t peerAddr[BDA_ADDR_LEN] = {0x12, 0x34, 0x56, 0x78, 0xAB, 0xFF};
uint8_t perDataBuf[ADV_BUF_LEN_MAX] = {0};

extAdvParam_t  extAdvParam[DM_NUM_ADV_SETS] =
{
  {
    {
      .priAdvInterMin  = 1000 / 0.625,      // 1s minimum interval
      .priAdvInterMax  = 1000 / 0.625,
      .priAdvChanMap   = DM_ADV_CHAN_ALL,
      .ownAddrType     = DM_ADDR_PUBLIC,
      .peerAddrType    = DM_ADDR_PUBLIC,
      .pPeerAddr       = peerAddr,
      .advFiltPolicy   = HCI_ADV_FILT_NONE,
      .advTxPwr        = HCI_TX_PWR_NO_PREFERENCE,
      .priAdvPhy       = HCI_ADV_PHY_LE_1M,
      .secAdvMaxSkip   = 0,
      .secAdvPhy       = HCI_ADV_PHY_LE_1M,
      .advSID          = 0,
      .scanReqNotifEna = FALSE,
    },
    .advType           = DM_ADV_CONN_UNDIRECT,
    .useLegAdvPdu      = TRUE,
  },

  {
      {
        .priAdvInterMin  = 1000 / 0.625,    // 1s minimum interval
        .priAdvInterMax  = 1000 / 0.625,
        .priAdvChanMap   = DM_ADV_CHAN_ALL,
        .ownAddrType     = DM_ADDR_PUBLIC,
        .peerAddrType    = DM_ADDR_PUBLIC,
        .pPeerAddr       = peerAddr,
        .advFiltPolicy   = HCI_ADV_FILT_NONE,
        .advTxPwr        = HCI_TX_PWR_NO_PREFERENCE,
        .priAdvPhy       = HCI_ADV_PHY_LE_1M,
        .secAdvMaxSkip   = 0,
        .secAdvPhy       = HCI_ADV_PHY_LE_2M,
        .advSID          = 1,
        .scanReqNotifEna = FALSE,
      },
      .advType           = DM_EXT_ADV_CONN_UNDIRECT,    //DM_ADV_CONN_UNDIRECT,
      .useLegAdvPdu      = FALSE,
  }
};


/*************************************************************************************************/
/*!
 *  \brief  Application DM callback.
 *
 *  \param  pDmEvt  DM callback event
 *
 *  \return None.
 */
/*************************************************************************************************/
static void peripheralPastDmCback(dmEvt_t *pDmEvt)
{
  dmEvt_t *pMsg;
  uint16_t len;

  len = DmSizeOfEvt(pDmEvt);

  if ((pMsg = WsfMsgAlloc(len)) != NULL)
  {
    memcpy((uint8_t *)pMsg, (uint8_t *)pDmEvt, len);
    WsfMsgSend(peripheralPastHandlerId, pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Application ATT callback.
 *
 *  \param  pEvt    ATT callback event
 *
 *  \return None.
 */
/*************************************************************************************************/
static void peripheralPastAttCback(attEvt_t *pEvt)
{
  attEvt_t *pMsg;

  if ((pMsg = WsfMsgAlloc(sizeof(attEvt_t) + pEvt->valueLen)) != NULL)
  {
    memcpy((uint8_t *)pMsg, (uint8_t *)pEvt, sizeof(attEvt_t));
    pMsg->pValue = (uint8_t *) (pMsg + 1);
    memcpy(pMsg->pValue, pEvt->pValue, pEvt->valueLen);
    WsfMsgSend(peripheralPastHandlerId, pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Application ATTS client characteristic configuration callback.
 *
 *  \param  pDmEvt  DM callback event
 *
 *  \return None.
 */
/*************************************************************************************************/
static void peripheralPastCccCback(attsCccEvt_t *pEvt)
{
  attsCccEvt_t  *pMsg;
  appDbHdl_t    dbHdl;

  /* If CCC not set from initialization and there's a device record and currently bonded */
  if ((pEvt->handle != ATT_HANDLE_NONE) &&
      ((dbHdl = AppDbGetHdl((dmConnId_t) pEvt->hdr.param)) != APP_DB_HDL_NONE) &&
      AppCheckBonded((dmConnId_t)pEvt->hdr.param))
  {
    /* Store value in device database. */
    AppDbSetCccTblValue(dbHdl, pEvt->idx, pEvt->value);
  }

  if ((pMsg = WsfMsgAlloc(sizeof(attsCccEvt_t))) != NULL)
  {
    memcpy((uint8_t *)pMsg, (uint8_t *)pEvt, sizeof(attsCccEvt_t));
    WsfMsgSend(peripheralPastHandlerId, pMsg);
  }
}


/*************************************************************************************************/
/*!
*  \brief  Send a Running Speed and Cadence Measurement Notification.
*
*  \param  connId    connection ID
*
*  \return None.
*/
/*************************************************************************************************/
static void peripheralPastSendRunningSpeedMeasurement(dmConnId_t connId)
{
  if (AttsCccEnabled(connId, PERIPHERAL_PAST_RSCS_SM_CCC_IDX))
  {
    static uint8_t walk_run = 1;

    RscpsSetParameter(RSCP_SM_PARAM_SPEED, 1);
    RscpsSetParameter(RSCP_SM_PARAM_CADENCE, 2);
    RscpsSetParameter(RSCP_SM_PARAM_STRIDE_LENGTH, 3);
    RscpsSetParameter(RSCP_SM_PARAM_TOTAL_DISTANCE, 4);

    /* Toggle running/walking */
    walk_run = walk_run? 0 : 1;
    RscpsSetParameter(RSCP_SM_PARAM_STATUS, walk_run);

    RscpsSendSpeedMeasurement(connId);
  }

  /* Configure and start timer to send the next measurement */
  peripheralPastRscmTimer.msg.event = PERIPHERAL_PAST_RUNNING_TIMER_IND;
  peripheralPastRscmTimer.msg.status = PERIPHERAL_PAST_RSCS_SM_CCC_IDX;
  peripheralPastRscmTimer.handlerId = peripheralPastHandlerId;
  peripheralPastRscmTimer.msg.param = connId;

  WsfTimerStartSec(&peripheralPastRscmTimer, peripheralPastRscmPeriod);
}

/*************************************************************************************************/
/*!
 *  \brief  Process CCC state change.
 *
 *  \param  pMsg    Pointer to message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void peripheralPastProcCccState(peripheralPastMsg_t *pMsg)
{
  APP_TRACE_INFO3("ccc state ind value:%d handle:%d idx:%d", pMsg->ccc.value, pMsg->ccc.handle, pMsg->ccc.idx);

  /* handle heart rate measurement CCC */
  if (pMsg->ccc.idx == PERIPHERAL_PAST_HRS_HRM_CCC_IDX)
  {
    if (pMsg->ccc.value == ATT_CLIENT_CFG_NOTIFY)
    {
      HrpsMeasStart((dmConnId_t) pMsg->ccc.hdr.param, PERIPHERAL_PAST_HR_TIMER_IND, PERIPHERAL_PAST_HRS_HRM_CCC_IDX);
    }
    else
    {
      HrpsMeasStop((dmConnId_t) pMsg->ccc.hdr.param);
    }
    return;
  }

  /* handle running speed and cadence measurement CCC */
  if (pMsg->ccc.idx == PERIPHERAL_PAST_RSCS_SM_CCC_IDX)
  {
    if (pMsg->ccc.value == ATT_CLIENT_CFG_NOTIFY)
    {
      peripheralPastSendRunningSpeedMeasurement((dmConnId_t)pMsg->ccc.hdr.param);
    }
    else
    {
      WsfTimerStop(&peripheralPastRscmTimer);
    }
    return;
  }

  /* handle battery level CCC */
  if (pMsg->ccc.idx == PERIPHERAL_PAST_BATT_LVL_CCC_IDX)
  {
    if (pMsg->ccc.value == ATT_CLIENT_CFG_NOTIFY)
    {
      BasMeasBattStart((dmConnId_t) pMsg->ccc.hdr.param, PERIPHERAL_PAST_BATT_TIMER_IND, PERIPHERAL_PAST_BATT_LVL_CCC_IDX);
    }
    else
    {
      BasMeasBattStop((dmConnId_t) pMsg->ccc.hdr.param);
    }
    return;
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Perform UI actions on connection close.
 *
 *  \param  pMsg    Pointer to message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void peripheralPastClose(peripheralPastMsg_t *pMsg)
{
  /* stop heart rate measurement */
  HrpsMeasStop((dmConnId_t) pMsg->hdr.param);

  /* stop battery measurement */
  BasMeasBattStop((dmConnId_t) pMsg->hdr.param);

  /* Stop running speed and cadence timer */
  WsfTimerStop(&peripheralPastRscmTimer);
}


/*************************************************************************************************/
/*!
 *  \brief  Set up advertising parameters used in Extended Advertising
 *
 *  \param  NULL.
 *
 *  \return None.
 */
/*************************************************************************************************/
void AppExtSetParam(void)
{
    uint8_t advHandle = 0;

    for ( advHandle = 0; advHandle < DM_NUM_ADV_SETS; advHandle++ )
    {
        // set extended advertising interval
        peripheralPastAdvCfg.advInterval[advHandle] = (uint16_t)(extAdvParam[advHandle].advParam.priAdvInterMin);

        DmAdvSetChannelMap(advHandle, extAdvParam[advHandle].advParam.priAdvChanMap);

        DmAdvSetAddrType(extAdvParam[advHandle].advParam.ownAddrType);

        AppExtSetAdvPeerAddr(advHandle, extAdvParam[advHandle].advParam.peerAddrType,
                            extAdvParam[advHandle].advParam.pPeerAddr);

        DmDevSetExtFilterPolicy(advHandle, DM_FILT_POLICY_MODE_ADV, HCI_ADV_FILT_NONE);

        DmAdvIncTxPwr(advHandle, FALSE, extAdvParam[advHandle].advParam.advTxPwr);


#if (BT_54)
#if defined(BLE_CSSA)
        DmAdvSetPhyParamV2(advHandle, extAdvParam[advHandle].advParam.priAdvPhy,
                            extAdvParam[advHandle].advParam.secAdvMaxSkip, extAdvParam[advHandle].advParam.secAdvPhy,
                            extAdvParam[advHandle].advParam.priAdvPhyOpt, extAdvParam[advHandle].advParam.secAdvPhyOpt);
#else
        DmAdvSetPhyParam(advHandle, extAdvParam[advHandle].advParam.priAdvPhy,
                            extAdvParam[advHandle].advParam.secAdvMaxSkip, extAdvParam[advHandle].advParam.secAdvPhy);
#endif // defined(BLE_CSSA)
        DmAdvSetPhyParam(advHandle, extAdvParam[advHandle].advParam.priAdvPhy,
                            extAdvParam[advHandle].advParam.secAdvMaxSkip, extAdvParam[advHandle].advParam.secAdvPhy);
#endif // BT_54

        DmAdvScanReqNotifEnable(advHandle, extAdvParam[advHandle].advParam.scanReqNotifEna);

        AppExtSetAdvType(advHandle, extAdvParam[advHandle].advType);

        DmAdvUseLegacyPdu(advHandle, extAdvParam[advHandle].useLegAdvPdu);
        peripheralPastAdvCfg.useLegacyPdu[advHandle] = extAdvParam[advHandle].useLegAdvPdu;
    }
}

uint8_t antena_id[HCI_MIN_NUM_ANTENNA_IDS] = {0, 1};

/*************************************************************************************************/
/*!
 *  \brief  Set up advertising and other procedures that need to be performed after
 *          device reset.
 *
 *  \param  pMsg    Pointer to message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void peripheralPastSetup(peripheralPastMsg_t *pMsg)
{
  AppExtSetParam();

  AppExtAdvSetData(adv_handle[0], APP_ADV_DATA_DISCOVERABLE, sizeof(peripheralPastAdvDataDisc), (uint8_t *) peripheralPastAdvDataDisc, ADV_BUF_LEN_MAX);
  AppExtAdvSetData(adv_handle[0], APP_SCAN_DATA_DISCOVERABLE, sizeof(peripheralPastScanDataDisc), (uint8_t *) peripheralPastScanDataDisc, ADV_BUF_LEN_MAX);

  AppExtAdvStart(1, adv_handle, APP_MODE_AUTO_INIT);
}

/*************************************************************************************************/
/*!
 *  \brief  Process messages from the event handler.
 *
 *  \param  pMsg    Pointer to message.
 *
 *  \return None.
 */
/*************************************************************************************************/
static void peripheralPastProcMsg(peripheralPastMsg_t *pMsg)
{
  uint8_t uiEvent = APP_UI_NONE;
  uint8_t i = 0;

  switch(pMsg->hdr.event)
  {
    case PERIPHERAL_PAST_RUNNING_TIMER_IND:
      peripheralPastSendRunningSpeedMeasurement((dmConnId_t)pMsg->ccc.hdr.param);
      break;

    case PERIPHERAL_PAST_HR_TIMER_IND:
      HrpsProcMsg(&pMsg->hdr);
      break;

    case PERIPHERAL_PAST_BATT_TIMER_IND:
      BasProcMsg(&pMsg->hdr);
      break;

    case ATTS_HANDLE_VALUE_CNF:
      HrpsProcMsg(&pMsg->hdr);
      BasProcMsg(&pMsg->hdr);
      break;

    case ATTS_CCC_STATE_IND:
      peripheralPastProcCccState(pMsg);
      break;

    case ATT_MTU_UPDATE_IND:
      APP_TRACE_INFO1("Negotiated MTU %d", ((attEvt_t *)pMsg)->mtu);
      break;

    case DM_RESET_CMPL_IND:
      // set database hash calculating status to true until a new hash is generated after reset
      attsCsfSetHashUpdateStatus(TRUE);

      // Generate ECC key if configured support secure connection,
      // else will calcualte ATT database hash
      if ( peripheralPastSecCfg.auth & DM_AUTH_SC_FLAG )
      {
          DmSecGenerateEccKeyReq();
      }
      else
      {
          AttsCalculateDbHash();
      }

      uiEvent = APP_UI_RESET_CMPL;
      break;

    case ATTS_DB_HASH_CALC_CMPL_IND:
      peripheralPastSetup(pMsg);
      break;

    case DM_ADV_SET_START_IND:
      uiEvent = APP_UI_ADV_SET_START_IND;
      break;

    case DM_ADV_SET_STOP_IND:
    {
        for ( i = 0; i< DM_CONN_MAX; i++ )
        {
          if ( !appExtConnCb[i].used )
          {
              appExtConnCb[i].used = TRUE;
              appExtConnCb[i].advHandle = pMsg->dm.advSetStop.advHandle;
              appExtConnCb[i].connHandle = pMsg->dm.advSetStop.handle;

              break;
          }
        }

        uiEvent = APP_UI_ADV_SET_STOP_IND;
    }
    break;

    case DM_ADV_START_IND:
        uiEvent = APP_UI_ADV_START;
    break;

    case DM_ADV_STOP_IND:
      uiEvent = APP_UI_ADV_STOP;
      break;

    case DM_CONN_OPEN_IND:
      HrpsProcMsg(&pMsg->hdr);
      BasProcMsg(&pMsg->hdr);
      // AppSlaveSecurityReq(1);
      uiEvent = APP_UI_CONN_OPEN;
#if (BT_53)
#if defined(BLE_PAST)
      uint16_t skip = 1;
      uint16_t sync_timeout = 0x03e8; // 10ms unit
      uint8_t cte_type = 0; // HCI_LE_SET_PER_ADV_NOT_SYNC_ALL;

      DmPastConfig(1, HCI_SYNC_TRSF_MODE_REP_DUP_FILT_ENABLE, skip, sync_timeout, cte_type);
#endif // defined(BLE_PAST)
#endif // BT_53
      break;

    case DM_CONN_CLOSE_IND:
    {
        uint8_t connHdl = pMsg->dm.connClose.handle;

        for ( i = 0; i< DM_CONN_MAX; i++ )
        {
            if ( appExtConnCb[i].used && (connHdl == appExtConnCb[i].connHandle) )
            {
                appExtConnCb[i].used = FALSE;
                AppExtAdvStart(1, &appExtConnCb[i].advHandle, APP_MODE_AUTO_INIT);

                break;
            }
        }

        peripheralPastClose(pMsg);
        uiEvent = APP_UI_CONN_CLOSE;
    }
    break;

    case DM_PHY_UPDATE_IND:
      APP_TRACE_INFO3("DM_PHY_UPDATE_IND status: %d, RX: %d, TX: %d", pMsg->dm.phyUpdate.status, pMsg->dm.phyUpdate.rxPhy, pMsg->dm.phyUpdate.txPhy);
      break;

    case DM_SEC_PAIR_CMPL_IND:
      DmSecGenerateEccKeyReq();
      uiEvent = APP_UI_SEC_PAIR_CMPL;
      break;

    case DM_SEC_PAIR_FAIL_IND:
      DmSecGenerateEccKeyReq();
      uiEvent = APP_UI_SEC_PAIR_FAIL;
      break;

    case DM_SEC_ENCRYPT_IND:
      uiEvent = APP_UI_SEC_ENCRYPT;
      break;

    case DM_SEC_ENCRYPT_FAIL_IND:
      uiEvent = APP_UI_SEC_ENCRYPT_FAIL;
      break;

    case DM_SEC_AUTH_REQ_IND:
      AppHandlePasskey(&pMsg->dm.authReq);
      break;

    case DM_SEC_ECC_KEY_IND:
      DmSecSetEccKey(&pMsg->dm.eccMsg.data.key);
      // Only calculate database hash if the calculating status is in progress
      if ( attsCsfGetHashUpdateStatus() )
      {
        AttsCalculateDbHash();
      }
      break;

    case DM_SEC_COMPARE_IND:
      AppHandleNumericComparison(&pMsg->dm.cnfInd);
      break;

    case DM_PRIV_CLEAR_RES_LIST_IND:
      APP_TRACE_INFO1("Clear resolving list status 0x%02x", pMsg->hdr.status);
      break;

#if (BT_53)
#if defined(BLE_PAST)
    case DM_PER_ADV_SYNC_TRSF_EST_IND:
    {
      uint8_t sync_id = pMsg->dm.perAdvSyncTrsfEst.advSid;

      APP_TRACE_INFO1("Periodic advertising sync transfer established, sync id: %d", sync_id);
      DmPastRptRcvEnable(sync_id, TRUE);
    }
      break;

    case DM_PER_ADV_REPORT_IND:
    {
      if (*(pMsg->dm.perAdvReport.pData + 1) == DM_ADV_TYPE_LOCAL_NAME)
      {
        char rx_adv_data[128] = {0};
        memcpy(rx_adv_data, (char *)(pMsg->dm.perAdvReport.pData + 2), pMsg->dm.perAdvReport.len - 2);
        APP_TRACE_INFO2("[PER_ADV_REPORT] RSSI: %d, data: %s", pMsg->dm.perAdvReport.rssi, rx_adv_data);
      }
      else
      {
        APP_TRACE_INFO2("[PER_ADV_REPORT] RSSI: %d, data length: %d", pMsg->dm.perAdvReport.rssi, pMsg->dm.perAdvReport.len);
      }
    }
      break;
#endif // defined(BLE_PAST)
#endif // BT_53

    case DM_HW_ERROR_IND:
      uiEvent = APP_UI_HW_ERROR;
      break;

    case DM_VENDOR_SPEC_CMD_CMPL_IND:
      break;

    default:
      break;
  }

  if (uiEvent != APP_UI_NONE)
  {
    AppUiAction(uiEvent);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Application handler init function called during system initialization.
 *
 *  \param  handlerID  WSF handler ID.
 *
 *  \return None.
 */
/*************************************************************************************************/
void PeripheralPastHandlerInit(wsfHandlerId_t handlerId)
{
  /* store handler ID */
  peripheralPastHandlerId = handlerId;

  /* Set configuration pointers */
  pAppExtAdvCfg = (appExtAdvCfg_t*)&peripheralPastAdvCfg;
  pAppSlaveCfg = (appSlaveCfg_t *) &peripheralPastSlaveCfg;
  pAppSecCfg = (appSecCfg_t *) &peripheralPastSecCfg;
  pAppUpdateCfg = (appUpdateCfg_t *) &peripheralPastUpdateCfg;

  /* Initialize application framework */
  AppSlaveInit();
  AppServerInit();

  /* Set stack configuration pointers */
  pSmpCfg = (smpCfg_t *) &peripheralPastSmpCfg;

  /* initialize heart rate profile sensor */
  HrpsInit(handlerId, (hrpsCfg_t *) &peripheralPastHrpsCfg);
  HrpsSetFlags(peripheralPastHrmFlags);

  /* initialize battery service server */
  BasInit(handlerId, (basCfg_t *) &peripheralPastBasCfg);
}

/*************************************************************************************************/
/*!
 *  \brief  WSF event handler for application.
 *
 *  \param  event   WSF event mask.
 *  \param  pMsg    WSF message.
 *
 *  \return None.
 */
/*************************************************************************************************/
void PeripheralPastHandler(wsfEventMask_t event, wsfMsgHdr_t *pMsg)
{
  if (pMsg != NULL)
  {
    /* process ATT messages */
    if (pMsg->event >= ATT_CBACK_START && pMsg->event <= ATT_CBACK_END)
    {
      /* process server-related ATT messages */
      AppServerProcAttMsg(pMsg);
    }
    /* process DM messages */
    else if (pMsg->event >= DM_CBACK_START && pMsg->event <= DM_CBACK_END)
    {
      /* process advertising and connection-related messages */
      AppSlaveProcDmMsg((dmEvt_t *) pMsg);

      /* process security-related messages */
      AppSlaveSecProcDmMsg((dmEvt_t *) pMsg);
    }

    /* perform profile and user interface-related operations */
    peripheralPastProcMsg((peripheralPastMsg_t *) pMsg);
  }
}

/*************************************************************************************************/
/*!
 *  \brief  Start the application.
 *
 *  \return None.
 */
/*************************************************************************************************/
void PeripheralPastStart(void)
{
  /* Register for stack callbacks */
  DmRegister(peripheralPastDmCback);
  DmConnRegister(DM_CLIENT_ID_APP, peripheralPastDmCback);
  AttRegister(peripheralPastAttCback);
  AttConnRegister(AppServerConnCback);
  AttsCccRegister(PERIPHERAL_PAST_NUM_CCC_IDX, (attsCccSet_t *) peripheralPastCccSet, peripheralPastCccCback);

  /* Initialize attribute server database */
  SvcCoreGattCbackRegister(GattReadCback, GattWriteCback);
  SvcCoreAddGroup();
  SvcHrsCbackRegister(NULL, HrpsWriteCback);
  SvcHrsAddGroup();
  SvcDisAddGroup();
  SvcBattCbackRegister(BasReadCback, NULL);
  SvcBattAddGroup();
  SvcRscsAddGroup();

  /* Set Service Changed CCCD index. */
  GattSetSvcChangedIdx(PERIPHERAL_PAST_GATT_SC_CCC_IDX);

  /* Set running speed and cadence features */
  RscpsSetFeatures(RSCS_ALL_FEATURES);

  /* Reset the device */
  DmDevReset();
}
