//*****************************************************************************
//
//! @file ble_freertos_periodic_adv.c
//!
//! @brief Ambiq Micro's demonstration of Periodic Advertising application.
//!
//! @addtogroup ble_examples BLE Examples
//!
//! @defgroup ble_freertos_periodic_adv BLE FreeRTOS Periodic Advertising Example
//! @ingroup ble_examples
//! @{
//!
//! Purpose: This example demonstrates BLE (Bluetooth Low Energy)
//! periodic advertising functionality within the FreeRTOS framework. The
//! application showcases data broadcasting capabilities for sensor
//! data and status information with intelligent power management. The example
//! provides a foundation for BLE applications requiring efficient data transmission.
//!
//! @section ble_freertos_periodic_adv_features Key Features
//!
//! 1. @b Periodic @b Advertising: Implements BLE periodic advertising
//!    functionality for efficient data transmission
//!
//! 2. @b FreeRTOS @b Integration: Utilizes FreeRTOS real-time operating system
//!    for reliable task scheduling and power management
//!
//! 3. @b Power @b Optimization: Minimizes power usage with intelligent
//!    advertising intervals and power management
//!
//! 4. @b Data @b Broadcasting: Provides efficient data broadcasting capabilities
//!    for sensor data and status information
//!
//! 5. @b Low @b Power @b Design: Optimized for deep sleep operation with
//!    intelligent power management for battery-powered devices
//!
//! @section ble_freertos_periodic_adv_functionality Functionality
//!
//! The application performs the following operations:
//! - Initializes BLE periodic advertising within FreeRTOS framework
//! - Implements efficient data broadcasting for sensor data and status
//! - Provides power optimization with intelligent advertising intervals
//! - Manages deep sleep operation for battery-powered devices
//! - Supports periodic advertising functionality
//! - Implements BLE periodic advertising features
//!
//! @section ble_freertos_periodic_adv_usage Usage
//!
//! 1. Compile and download the application to target device
//! 2. Optionally enable debug printing with AM_DEBUG_PRINTF
//! 3. Monitor ITM output for periodic advertising operations
//! 4. Test data broadcasting and power optimization
//! 5. Verify deep sleep operation and battery efficiency
//!
//! @section ble_freertos_periodic_adv_configuration Configuration
//!
//! - @b AM_DEBUG_PRINTF: Enable debug printing (optional)
//! - @b ITM: Output for debug messages (1MHz)
//! - @b Periodic @b Advertising: BLE periodic advertising configuration
//! - @b Power @b Management: Deep sleep optimization for battery-powered devices
//
//*****************************************************************************

//*****************************************************************************
//
// Copyright (c) 2026, Ambiq Micro, Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the copyright holder nor the names of its
// contributors may be used to endorse or promote products derived from this
// software without specific prior written permission.
//
// Third party software included in this distribution is subject to the
// additional license terms as defined in the /docs/licenses directory.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// This is part of revision stable-e24d618f43 of the AmbiqSuite Development Package.
//
//*****************************************************************************
#include "ble_freertos_periodic_adv.h"
#include "rtos.h"

#if defined(AM_PART_APOLLO330P_510L)
#include "am_devices_510L_radio.h"
#endif // AM_PART_APOLLO330P_510L

//*****************************************************************************
//
// Global Variables
//
//*****************************************************************************
#if defined(AM_PART_APOLLO330P_510L)
// Allocate the non-cached IPC share memory from SSRAM
AM_SHARED_RW uint32_t g_pui32IpcShm[AM_DEVICES_510L_RADIO_IPC_SHM_DEFAULT_SIZE / sizeof(uint32_t)] __attribute__((aligned(32)));

am_hal_mpu_region_config_t sMPUCfg =
{
    .ui32RegionNumber = 6,
    .ui32BaseAddress = (uint32_t)g_pui32IpcShm,
    .eShareable = NON_SHARE,
    .eAccessPermission = RW_NONPRIV,
    .bExecuteNever = true,
    .ui32LimitAddress = (uint32_t)g_pui32IpcShm + sizeof(g_pui32IpcShm) - 1,
    .ui32AttrIndex = 0,
    .bEnable = true,
};

am_hal_mpu_attr_t sMPUAttr =
{
    .ui8AttrIndex = 0,
    .bNormalMem = true,
    .sOuterAttr = {
                    .bNonTransient = false,
                    .bWriteBack = true,
                    .bReadAllocate = false,
                    .bWriteAllocate = false
                  },
    .sInnerAttr = {
                    .bNonTransient = false,
                    .bWriteBack = true,
                    .bReadAllocate = false,
                    .bWriteAllocate = false
                  },
    .eDeviceAttr = DEV_nGnRnE,
};

am_devices_510L_radio_ipc_shm_t sIpcShm =
{
    .ui32IpcShmAddr = (uint32_t)g_pui32IpcShm,
    .ui32IpcShmSize = sizeof(g_pui32IpcShm)
};
#endif // AM_PART_APOLLO330P_510L

//*****************************************************************************
//
// Enable printing to the console.
//
//*****************************************************************************
void
enable_print_interface(void)
{
    //
    // Initialize a debug printing interface.
    //
    am_bsp_itm_printf_enable();

    //
    // Print the banner.
    //
    am_util_stdio_terminal_clear();

}

//*****************************************************************************
//
// Main Function
//
//*****************************************************************************
int
main(void)
{
    //
    // Set the clock frequency
    //
#ifndef NOFPU
    //
    // Enable the floating point module, and configure the core for lazy
    // stacking.
    //
    am_hal_sysctrl_fpu_enable();
    am_hal_sysctrl_fpu_stacking_enable(true);
#else
    am_hal_sysctrl_fpu_disable();
#endif
    //
    // Configure the board for low power.
    //
    am_bsp_low_power_init();

#if defined(AM_PART_APOLLO330P_510L)
    //
    // Set up the attributes.
    //
    am_hal_mpu_attr_configure(&sMPUAttr, 1);
    //
    // Clear the MPU regions.
    //
    am_hal_mpu_region_clear();
    //
    // Set up the regions.
    //
    am_hal_mpu_region_configure(&sMPUCfg, 1);
    //
    // Invalidate and clear DCACHE, this is required by CM55 TRF.
    //
    am_hal_cachectrl_dcache_invalidate(NULL, true);
    //
    // MPU enable
    //
    am_hal_mpu_enable(true, true);
    //
    // Initialize the IPC share memory
    //
    am_devices_510L_radio_ipc_shm_init(&sIpcShm);
#endif // AM_PART_APOLLO330P_510L

    //
    // Enable printing to the console.
    //
#ifdef AM_DEBUG_PRINTF
    enable_print_interface();
#endif

    //
    // Initialize plotting interface.
    //
    am_util_debug_printf("FreeRTOS Periodic Advertising Example\n");

    //
    // Run the application.
    //
    run_tasks();

    //
    // We shouldn't ever get here.
    //
    while (1)
    {
    }
}

//*****************************************************************************
//
// End Doxygen group.
//! @}
//
//*****************************************************************************
